/*
 * This file is part of the Score-P software (http://www.score-p.org)
 *
 * Copyright (c) 2009-2013, 2019,
 * RWTH Aachen University, Germany
 *
 * Copyright (c) 2009-2013,
 * Gesellschaft fuer numerische Simulation mbH Braunschweig, Germany
 *
 * Copyright (c) 2009-2019,
 * Technische Universitaet Dresden, Germany
 *
 * Copyright (c) 2009-2013,
 * University of Oregon, Eugene, USA
 *
 * Copyright (c) 2009-2019,
 * Forschungszentrum Juelich GmbH, Germany
 *
 * Copyright (c) 2009-2014,
 * German Research School for Simulation Sciences GmbH, Juelich/Aachen, Germany
 *
 * Copyright (c) 2009-2013,
 * Technische Universitaet Muenchen, Germany
 *
 * Copyright (c) 2016,
 * Technische Universitaet Darmstadt, Germany
 *
 * This software may be modified and distributed under the terms of
 * a BSD-style license.  See the COPYING file in the package base
 * directory for details.
 *
 */

/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2011                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2010-2011                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2003-2008                                                **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/


/**
 * @file
 * @ingroup    MPI_Wrapper
 *
 * @brief C interface wrappers for one-sided communication
 */

#include <config.h>

#include "SCOREP_Mpi.h"
#include "scorep_mpi_communicator.h"
#include "scorep_mpi_rma_request.h"
#include <SCOREP_RuntimeManagement.h>
#include <SCOREP_InMeasurement.h>
#include <SCOREP_Events.h>
#include <SCOREP_AllocMetric.h>

/** internal id counter for rma operations */
static int scorep_rma_id = 0;

/**
 * @name C wrappers for access functions
 * @{
 */

#if HAVE( DECL_PMPI_ACCUMULATE ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Accumulate )
/**
 * Measurement wrapper for MPI_Accumulate
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaAtomic.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers three events: Enter, RmaAtomic, and Leave.
 */
int
MPI_Accumulate( SCOREP_MPI_CONST_DECL void* origin_addr, int origin_count, MPI_Datatype origin_datatype, int target_rank, MPI_Aint target_disp, int target_count, MPI_Datatype target_datatype, MPI_Op op, MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ACCUMULATE ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                uint64_t bytes_get = 0;
                uint64_t bytes_put = origin_count;

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // is explicit request?
                  // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaAtomic( win_handle, target_rank,
                                  SCOREP_RMA_ATOMIC_TYPE_ACCUMULATE,
                                  bytes_put, bytes_get, matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ACCUMULATE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Accumulate( origin_addr, origin_count, origin_datatype, target_rank, target_disp, target_count, target_datatype, op, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // rma_explicit_handle
                  // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                // Only implicit request tracking needed
                if ( rma_request == NULL )
                {
                    // Create a new request for key tuple
                    // (win, target, completion)
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ACCUMULATE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ACCUMULATE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_COMPARE_AND_SWAP ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Compare_and_swap ) && defined( SCOREP_MPI_COMPARE_AND_SWAP_PROTO_ARGS )
/**
 * Measurement wrapper for MPI_Compare_and_swap
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaAtomicWithProto.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers three events: Enter, RmaAtomic, and Leave.
 */
int
MPI_Compare_and_swap SCOREP_MPI_COMPARE_AND_SWAP_PROTO_ARGS
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_COMPARE_AND_SWAP ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                uint64_t bytes_get = 1;
                uint64_t bytes_put = 1;

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // is explicit request?
                  // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaAtomic( win_handle, target_rank,
                                  SCOREP_RMA_ATOMIC_TYPE_COMPARE_AND_SWAP,
                                  bytes_put, bytes_get, matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_COMPARE_AND_SWAP ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Compare_and_swap( origin_addr, compare_addr, result_addr, datatype, target_rank, target_disp, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // rma_explicit_handle
                  // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                // Create a new request for key tuple (win, target, completion)
                if ( rma_request == NULL )
                {
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_COMPARE_AND_SWAP ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_COMPARE_AND_SWAP ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_FETCH_AND_OP ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Fetch_and_op ) && defined( SCOREP_MPI_FETCH_AND_OP_PROTO_ARGS )
/**
 * Measurement wrapper for MPI_Fetch_and_op
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaAtomicWithProto.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers three events: Enter, RmaAtomic, and Leave.
 */
int
MPI_Fetch_and_op SCOREP_MPI_FETCH_AND_OP_PROTO_ARGS
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_FETCH_AND_OP ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                uint64_t bytes_get = 1;
                uint64_t bytes_put = 1;

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // is explicit request?
                  // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaAtomic( win_handle, target_rank,
                                  SCOREP_RMA_ATOMIC_TYPE_FETCH_AND_ACCUMULATE,
                                  bytes_put, bytes_get, matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_FETCH_AND_OP ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Fetch_and_op( origin_addr, result_addr, datatype, target_rank, target_disp, op, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // rma_explicit_handle
                  // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                // Create a new request for key tuple (win, target, completion)
                if ( rma_request == NULL )
                {
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_FETCH_AND_OP ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_FETCH_AND_OP ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_GET ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Get )
/**
 * Measurement wrapper for MPI_Get
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaGet.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers three events: Enter, RmaGet, and Leave.
 */
int
MPI_Get( void* origin_addr, int origin_count, MPI_Datatype origin_datatype, int target_rank, MPI_Aint target_disp, int target_count, MPI_Datatype target_datatype, MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GET ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                int origin_datatype_size;
                PMPI_Type_size( origin_datatype, &origin_datatype_size );

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // is explicit request?
                  // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_COMBINED_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaGet( win_handle, target_rank,
                               origin_count * origin_datatype_size,
                               matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GET ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Get( origin_addr, origin_count, origin_datatype, target_rank, target_disp, target_count, target_datatype, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // rma_explicit_handle
                  // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_COMBINED_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                // Only implicit request tracking needed
                if ( rma_request == NULL )
                {
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_COMBINED_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GET ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GET ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_GET_ACCUMULATE ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Get_accumulate )
/**
 * Measurement wrapper for MPI_Get_accumulate
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaAtomic.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers three events: Enter, RmaAtomic, and Leave.
 */
int
MPI_Get_accumulate( SCOREP_MPI_CONST_DECL void* origin_addr, int origin_count, MPI_Datatype origin_datatype, void* result_addr, int result_count, MPI_Datatype result_datatype, int target_rank, MPI_Aint target_disp, int target_count, MPI_Datatype target_datatype, MPI_Op op, MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GET_ACCUMULATE ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                uint64_t bytes_get = result_count;
                uint64_t bytes_put = origin_count;

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // is explicit request?
                  // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaAtomic( win_handle, target_rank,
                                  SCOREP_RMA_ATOMIC_TYPE_FETCH_AND_ACCUMULATE,
                                  bytes_put, bytes_get, matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GET_ACCUMULATE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Get_accumulate( origin_addr, origin_count, origin_datatype, result_addr, result_count, result_datatype, target_rank, target_disp, target_count, target_datatype, op, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // rma_explicit_handle
                  // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                // Only implicit request tracking needed
                if ( rma_request == NULL )
                {
                    // Create a new request for key tuple
                    // (win, target, completion)
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GET_ACCUMULATE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GET_ACCUMULATE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_PUT ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Put )
/**
 * Measurement wrapper for MPI_Put
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaPut.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers three events: Enter, RmaPut, and Leave.
 */
int
MPI_Put( SCOREP_MPI_CONST_DECL void* origin_addr, int origin_count, MPI_Datatype origin_datatype, int target_rank, MPI_Aint target_disp, int target_count, MPI_Datatype target_datatype, MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_PUT ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                int origin_datatype_size;
                PMPI_Type_size( origin_datatype, &origin_datatype_size );

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // is explicit request?
                  // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaPut( scorep_mpi_win_handle( win ), target_rank,
                               ( uint64_t )origin_count * origin_datatype_size,
                               matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_PUT ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Put( origin_addr, origin_count, origin_datatype, target_rank, target_disp, target_count, target_datatype, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if false // rma_explicit_handle
                  // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                if ( rma_request == NULL )
                {
                    // Create a new request for key tuple
                    // (win, target, completion)
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_PUT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_PUT ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_RACCUMULATE ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Raccumulate ) && defined( SCOREP_MPI_RACCUMULATE_PROTO_ARGS )
/**
 * Measurement wrapper for MPI_Raccumulate
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaAtomicWithProto.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers three events: Enter, RmaAtomic, and Leave.
 */
int
MPI_Raccumulate SCOREP_MPI_RACCUMULATE_PROTO_ARGS
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RACCUMULATE ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                uint64_t bytes_get = 0;
                uint64_t bytes_put = origin_count;

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if true // is explicit request?
                 // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaAtomic( win_handle, target_rank,
                                  SCOREP_RMA_ATOMIC_TYPE_ACCUMULATE,
                                  bytes_put, bytes_get, matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RACCUMULATE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Raccumulate( origin_addr, origin_count, origin_datatype, target_rank, target_disp, target_count, target_datatype, op, win, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if true // rma_explicit_handle
                 // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                // Create a new request for key tuple (win, target, completion)
                if ( rma_request == NULL )
                {
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RACCUMULATE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RACCUMULATE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_RGET ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Rget )
/**
 * Measurement wrapper for MPI_Rget
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaGet.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers three events: Enter, RmaGet, and Leave.
 */
int
MPI_Rget( void* origin_addr, int origin_count, MPI_Datatype origin_datatype, int target_rank, MPI_Aint target_disp, int target_count, MPI_Datatype target_datatype, MPI_Win win, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RGET ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                int origin_datatype_size;
                PMPI_Type_size( origin_datatype, &origin_datatype_size );

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if true // is explicit request?
                 // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_COMBINED_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaGet( win_handle, target_rank,
                               origin_count * origin_datatype_size,
                               matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RGET ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Rget( origin_addr, origin_count, origin_datatype, target_rank, target_disp, target_count, target_datatype, win, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if true // rma_explicit_handle
                 // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_COMBINED_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                // Only implicit request tracking needed
                if ( rma_request == NULL )
                {
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_COMBINED_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RGET ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RGET ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_RGET_ACCUMULATE ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Rget_accumulate )
/**
 * Measurement wrapper for MPI_Rget_accumulate
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaAtomic.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers three events: Enter, RmaAtomic, and Leave.
 */
int
MPI_Rget_accumulate( SCOREP_MPI_CONST_DECL void* origin_addr, int origin_count, MPI_Datatype origin_datatype, void* result_addr, int result_count, MPI_Datatype result_datatype, int target_rank, MPI_Aint target_disp, int target_count, MPI_Datatype target_datatype, MPI_Op op, MPI_Win win, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RGET_ACCUMULATE ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                uint64_t bytes_get = result_count;
                uint64_t bytes_put = origin_count;

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if true // is explicit request?
                 // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaAtomic( win_handle, target_rank,
                                  SCOREP_RMA_ATOMIC_TYPE_FETCH_AND_ACCUMULATE,
                                  bytes_put, bytes_get, matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RGET_ACCUMULATE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Rget_accumulate( origin_addr, origin_count, origin_datatype, result_addr, result_count, result_datatype, target_rank, target_disp, target_count, target_datatype, op, win, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if true // rma_explicit_handle
                 // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                // Only implicit request tracking needed
                if ( rma_request == NULL )
                {
                    // Create a new request for key tuple
                    // (win, target, completion)
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RGET_ACCUMULATE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RGET_ACCUMULATE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_RPUT ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Rput )
/**
 * Measurement wrapper for MPI_Rput
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_RmaPut.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers three events: Enter, RmaPut, and Leave.
 */
int
MPI_Rput( SCOREP_MPI_CONST_DECL void* origin_addr, int origin_count, MPI_Datatype origin_datatype, int target_rank, MPI_Aint target_disp, int target_count, MPI_Datatype target_datatype, MPI_Win win, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int               event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int               event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    scorep_mpi_rma_request* rma_request                = NULL;
    SCOREP_RmaWindowHandle  win_handle                 = scorep_mpi_win_handle( win );
    SCOREP_MpiRequestId     matching_id                = 0;
    int                     return_val;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RPUT ] );
            if ( target_rank != MPI_PROC_NULL )
            {
                int origin_datatype_size;
                PMPI_Type_size( origin_datatype, &origin_datatype_size );

                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if true // is explicit request?
                 // Explicit request always get a new matching id
                matching_id = scorep_mpi_get_request_id();
        #else
                // Implicit request may already have a valid matching id
                rma_request = scorep_mpi_rma_request_find( win_handle,
                                                           target_rank,
                                                           MPI_REQUEST_NULL,
                                                           SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION );

                if ( rma_request == NULL )
                {
                    matching_id = scorep_mpi_get_request_id();
                }
                else
                {
                    matching_id = rma_request->matching_id;
                }
        #endif

                SCOREP_RmaPut( scorep_mpi_win_handle( win ), target_rank,
                               ( uint64_t )origin_count * origin_datatype_size,
                               matching_id );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RPUT ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Rput( origin_addr, origin_count, origin_datatype, target_rank, target_disp, target_count, target_datatype, win, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( target_rank != MPI_PROC_NULL )
            {
                /*
                 * The following code block is only used if the prototype
                 * definition specifies:
                 *    <attribute id="rma_explicit_handle">true</attribute>
                 */
        #if true // rma_explicit_handle
                 // Both implicit _and_ explicit request tracking needed.

                // Start implicit request tracking
                rma_request =
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   *request,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );

                // Start explicit request tracking
                scorep_mpi_request_win_create( *request, rma_request );
        #else
                if ( rma_request == NULL )
                {
                    // Create a new request for key tuple
                    // (win, target, completion)
                    scorep_mpi_rma_request_create( win_handle,
                                                   target_rank,
                                                   MPI_REQUEST_NULL,
                                                   SCOREP_MPI_RMA_REQUEST_SEPARATE_COMPLETION,
                                                   matching_id );
                }
        #endif
            }
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RPUT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RPUT ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 * @name C wrappers for window management functions
 * @{
 */

#if HAVE( DECL_PMPI_WIN_ALLOCATE ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_allocate )
/**
 * Measurement wrapper for MPI_Win_allocate
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_WinCreate.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma

 * Triggers five events: Enter, RmaCollectiveBegin, WinCreate,
 * RmaCollectiveEnd, and Leave.
 */
int
MPI_Win_allocate( MPI_Aint size, int disp_unit, MPI_Info info, MPI_Comm comm, void* baseptr, MPI_Win* win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    SCOREP_RmaWindowHandle win_handle = SCOREP_INVALID_RMA_WINDOW;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            if ( scorep_mpi_memory_recording )
            {
                uint64_t size_as_uint64 = size;
                SCOREP_AddAttribute( scorep_mpi_memory_alloc_size_attribute,
                                     &size_as_uint64 );
            }


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ALLOCATE ] );
            SCOREP_RmaCollectiveBegin();
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ALLOCATE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_allocate( size, disp_unit, info, comm, baseptr, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( *win != MPI_WIN_NULL )
            {
                win_handle = scorep_mpi_win_create( "MPI_Win_allocate window", *win, comm );
                SCOREP_RmaWinCreate( win_handle );
            }


            if ( scorep_mpi_memory_recording && size > 0 && MPI_SUCCESS == return_val )
            {
                SCOREP_AllocMetric_HandleAlloc( scorep_mpi_allocations_metric,
                                                ( uint64_t )( *( void** )baseptr ),
                                                size );
            }


            SCOREP_RmaCollectiveEnd( SCOREP_COLLECTIVE_CREATE_HANDLE_AND_ALLOCATE,
                                     SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                     win_handle, SCOREP_INVALID_ROOT_RANK, 0, 0 );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ALLOCATE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ALLOCATE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_ALLOCATE_SHARED ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_allocate_shared )
/**
 * Measurement wrapper for MPI_Win_allocate_shared
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_WinCreate.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma

 * Triggers five events: Enter, RmaCollectiveBegin, WinCreate,
 * RmaCollectiveEnd, and Leave.
 */
int
MPI_Win_allocate_shared( MPI_Aint size, int disp_unit, MPI_Info info, MPI_Comm comm, void* baseptr, MPI_Win* win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    SCOREP_RmaWindowHandle win_handle = SCOREP_INVALID_RMA_WINDOW;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            if ( scorep_mpi_memory_recording )
            {
                uint64_t size_as_uint64 = size;
                SCOREP_AddAttribute( scorep_mpi_memory_alloc_size_attribute,
                                     &size_as_uint64 );
            }


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ALLOCATE_SHARED ] );
            SCOREP_RmaCollectiveBegin();
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ALLOCATE_SHARED ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_allocate_shared( size, disp_unit, info, comm, baseptr, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( *win != MPI_WIN_NULL )
            {
                win_handle = scorep_mpi_win_create( "MPI_Win_allocate_shared window", *win, comm );
                SCOREP_RmaWinCreate( win_handle );
            }


            if ( scorep_mpi_memory_recording && size > 0 && MPI_SUCCESS == return_val )
            {
                SCOREP_AllocMetric_HandleAlloc( scorep_mpi_allocations_metric,
                                                ( uint64_t )( *( void** )baseptr ),
                                                size );
            }


            SCOREP_RmaCollectiveEnd( SCOREP_COLLECTIVE_CREATE_HANDLE_AND_ALLOCATE,
                                     SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                     win_handle, SCOREP_INVALID_ROOT_RANK, 0, 0 );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ALLOCATE_SHARED ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ALLOCATE_SHARED ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_ATTACH ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_attach )
/**
 * Measurement wrapper for MPI_Win_attach
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_attach call with enter and exit events.
 */
int
MPI_Win_attach( MPI_Win win, void* base, MPI_Aint size )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ATTACH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ATTACH ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_attach( win, base, size );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ATTACH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_ATTACH ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_CREATE ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_create )
/**
 * Measurement wrapper for MPI_Win_create
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_WinCreate.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma

 * Triggers five events: Enter, RmaCollectiveBegin, WinCreate,
 * RmaCollectiveEnd, and Leave.
 */
int
MPI_Win_create( void* base, MPI_Aint size, int disp_unit, MPI_Info info, MPI_Comm comm, MPI_Win* win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    SCOREP_RmaWindowHandle win_handle = SCOREP_INVALID_RMA_WINDOW;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE ] );
            SCOREP_RmaCollectiveBegin();
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_create( base, size, disp_unit, info, comm, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( *win != MPI_WIN_NULL )
            {
                win_handle = scorep_mpi_win_create( "MPI_Win_create window", *win, comm );
                SCOREP_RmaWinCreate( win_handle );
            }


            SCOREP_RmaCollectiveEnd( SCOREP_COLLECTIVE_CREATE_HANDLE,
                                     SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                     win_handle, SCOREP_INVALID_ROOT_RANK, 0, 0 );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_CREATE_DYNAMIC ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_create_dynamic )
/**
 * Measurement wrapper for MPI_Win_create_dynamic
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_WinCreate.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma

 * Triggers five events: Enter, RmaCollectiveBegin, WinCreate,
 * RmaCollectiveEnd, and Leave.
 */
int
MPI_Win_create_dynamic( MPI_Info info, MPI_Comm comm, MPI_Win* win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    SCOREP_RmaWindowHandle win_handle = SCOREP_INVALID_RMA_WINDOW;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_DYNAMIC ] );
            SCOREP_RmaCollectiveBegin();
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_DYNAMIC ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_create_dynamic( info, comm, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( *win != MPI_WIN_NULL )
            {
                win_handle = scorep_mpi_win_create( "MPI_Win_create_dynamic window", *win, comm );
                SCOREP_RmaWinCreate( win_handle );
            }


            SCOREP_RmaCollectiveEnd( SCOREP_COLLECTIVE_CREATE_HANDLE,
                                     SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                     win_handle, SCOREP_INVALID_ROOT_RANK, 0, 0 );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_DYNAMIC ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_DYNAMIC ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_DETACH ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_detach ) && defined( SCOREP_MPI_WIN_DETACH_PROTO_ARGS )
/**
 * Measurement wrapper for MPI_Win_detach
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_StdWithProto.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_detach call with enter and exit events.
 */
int
MPI_Win_detach SCOREP_MPI_WIN_DETACH_PROTO_ARGS
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_DETACH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_DETACH ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_detach( win, base );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_DETACH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_DETACH ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif


#if HAVE( DECL_PMPI_WIN_FREE ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_free
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_free( MPI_Win* win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    SCOREP_RmaWindowHandle win_handle;
    SCOREP_CollectiveType  collective_type = SCOREP_COLLECTIVE_DESTROY_HANDLE;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FREE ] );
            SCOREP_RmaCollectiveBegin();

      #if ( MPI_VERSION >= 3 )
            // determine collective type of operation
            int* create_kind = NULL;
            int  flag        = 0;
            PMPI_Win_get_attr( *win, MPI_WIN_CREATE_FLAVOR, &create_kind, &flag );
            if ( flag )
            {
                switch ( *create_kind )
                {
                    case MPI_WIN_FLAVOR_CREATE:
                    case MPI_WIN_FLAVOR_DYNAMIC:
                        collective_type = SCOREP_COLLECTIVE_DESTROY_HANDLE;
                        break;
                    case MPI_WIN_FLAVOR_ALLOCATE:
                    case MPI_WIN_FLAVOR_SHARED:
                        collective_type = SCOREP_COLLECTIVE_DESTROY_HANDLE_AND_DEALLOCATE;
                        break;
                }
            }
      #endif
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FREE ] );
        }
    }

    // save handle reference for RmaCollectiveEnd
    win_handle = scorep_mpi_win_handle( *win );
    // stop tracking window
    scorep_mpi_win_free( *win );

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_free( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_RmaWinDestroy( win_handle );
            SCOREP_RmaCollectiveEnd( collective_type,
                                     SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                     win_handle, SCOREP_INVALID_ROOT_RANK, 0, 0 );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FREE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FREE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_SHARED_QUERY ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_shared_query )
/**
 * Measurement wrapper for MPI_Win_shared_query
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_shared_query call with enter and exit events.
 */
int
MPI_Win_shared_query( MPI_Win win, int rank, MPI_Aint* size, int* disp_unit, void* baseptr )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SHARED_QUERY ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SHARED_QUERY ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_shared_query( win, rank, size, disp_unit, baseptr );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SHARED_QUERY ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SHARED_QUERY ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 * @name C wrappers for synchronization functions
 * @{
 */

#if HAVE( DECL_PMPI_WIN_COMPLETE ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_complete
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_complete( MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_COMPLETE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_COMPLETE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_complete( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_RmaWindowHandle win_handle = scorep_mpi_win_handle( win );

            scorep_mpi_rma_request_foreach_on_window( win_handle,
                                                      scorep_mpi_rma_request_write_standard_completion_and_remove );

            SCOREP_RmaGroupSync( SCOREP_RMA_SYNC_LEVEL_MEMORY |
                                 SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                 win_handle,
                                 scorep_mpi_epoch_get_group_handle( win, SCOREP_MPI_RMA_ACCESS_EPOCH ) );

            scorep_mpi_epoch_end( win, SCOREP_MPI_RMA_ACCESS_EPOCH );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_COMPLETE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_COMPLETE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_FENCE ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_fence
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_fence( int     assert,
               MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FENCE ] );
            SCOREP_RmaCollectiveBegin();
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FENCE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_fence( assert, win );
    SCOREP_EXIT_WRAPPED_REGION();


    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_RmaWindowHandle win_handle = scorep_mpi_win_handle( win );

            scorep_mpi_rma_request_foreach_on_window( win_handle,
                                                      scorep_mpi_rma_request_write_standard_completion_and_remove );

            // TODO: evaluation of assert could lead to be better
            //       value for the SYNC_LEVEL

            SCOREP_RmaCollectiveEnd( SCOREP_COLLECTIVE_BARRIER,
                                     SCOREP_RMA_SYNC_LEVEL_MEMORY |
                                     SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                     win_handle,
                                     SCOREP_INVALID_ROOT_RANK,
                                     0, 0 );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FENCE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FENCE ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_FLUSH ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_flush )
/**
 * Measurement wrapper for MPI_Win_flush
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_flush call with enter and exit events.
 */
int
MPI_Win_flush( int rank, MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_flush( rank, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            scorep_mpi_rma_request_foreach_to_target( scorep_mpi_win_handle( win ),
                                                      rank,
                                                      scorep_mpi_rma_request_write_full_completion );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_FLUSH_ALL ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_flush_all )
/**
 * Measurement wrapper for MPI_Win_flush_all
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_flush_all call with enter and exit events.
 */
int
MPI_Win_flush_all( MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_ALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_ALL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_flush_all( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            scorep_mpi_rma_request_foreach_on_window( scorep_mpi_win_handle( win ),
                                                      scorep_mpi_rma_request_write_full_completion );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_ALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_ALL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_FLUSH_LOCAL ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_flush_local )
/**
 * Measurement wrapper for MPI_Win_flush_local
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_flush_local call with enter and exit events.
 */
int
MPI_Win_flush_local( int rank, MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_LOCAL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_LOCAL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_flush_local( rank, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            scorep_mpi_rma_request_foreach_to_target( scorep_mpi_win_handle( win ),
                                                      rank,
                                                      scorep_mpi_rma_request_write_standard_completion );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_LOCAL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_LOCAL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_FLUSH_LOCAL_ALL ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_flush_local_all )
/**
 * Measurement wrapper for MPI_Win_flush_local_all
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_flush_local_all call with enter and exit events.
 */
int
MPI_Win_flush_local_all( MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_LOCAL_ALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_LOCAL_ALL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_flush_local_all( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            scorep_mpi_rma_request_foreach_on_window( scorep_mpi_win_handle( win ),
                                                      scorep_mpi_rma_request_write_standard_completion );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_LOCAL_ALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FLUSH_LOCAL_ALL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_LOCK ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_lock
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_lock( int     lock_type,
              int     rank,
              int     assert,
              MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_LOCK ] );
            SCOREP_RmaRequestLock( scorep_mpi_win_handle( win ), rank, 0,
                                   lock_type == MPI_LOCK_EXCLUSIVE ? SCOREP_LOCK_EXCLUSIVE : SCOREP_LOCK_SHARED );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_LOCK ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_lock( lock_type, rank, assert, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_LOCK ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_LOCK ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_LOCK_ALL ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_lock_all )
/**
 * Measurement wrapper for MPI_Win_lock_all
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 *
 * Triggers an enter event, a request-lock event, and an exit event.
 *
 */
int
MPI_Win_lock_all( int assert, MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_LOCK_ALL ] );
            SCOREP_RmaRequestLock( scorep_mpi_win_handle( win ), SCOREP_ALL_TARGET_RANKS, 0,
                                   SCOREP_LOCK_SHARED );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_LOCK_ALL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_lock_all( assert, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_LOCK_ALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_LOCK_ALL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_POST ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_post
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_post( MPI_Group group,
              int       assert,
              MPI_Win   win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_POST ] );
            scorep_mpi_epoch_start( win, group, SCOREP_MPI_RMA_EXPOSURE_EPOCH );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_POST ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_post( group, assert, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_RmaGroupSync( SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                 scorep_mpi_win_handle( win ),
                                 scorep_mpi_group_handle( group ) );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_POST ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_POST ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_START ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_start
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_start( MPI_Group group,
               int       assert,
               MPI_Win   win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_START ] );
            scorep_mpi_epoch_start( win, group, SCOREP_MPI_RMA_ACCESS_EPOCH );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_START ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_start( group, assert, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_RmaGroupSync( SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                 scorep_mpi_win_handle( win ),
                                 scorep_mpi_group_handle( group ) );

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_START ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_START ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_SYNC ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_sync )
/**
 * Measurement wrapper for MPI_Win_sync
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_sync call with enter and exit events.
 */
int
MPI_Win_sync( MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SYNC ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SYNC ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_sync( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SYNC ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SYNC ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_TEST ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_test
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_test( MPI_Win win,
              int*    flag )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_TEST ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_TEST ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_test( win, flag );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( *flag != 0 )
            {
                scorep_mpi_rma_request_foreach_on_window( scorep_mpi_win_handle( win ),
                                                          scorep_mpi_rma_request_write_standard_completion );

                SCOREP_RmaGroupSync( SCOREP_RMA_SYNC_LEVEL_MEMORY |
                                     SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                     scorep_mpi_win_handle( win ),
                                     scorep_mpi_epoch_get_group_handle( win, SCOREP_MPI_RMA_EXPOSURE_EPOCH ) );
                scorep_mpi_epoch_end( win, SCOREP_MPI_RMA_EXPOSURE_EPOCH );
            }
            else
            {
                SCOREP_RmaGroupSync( SCOREP_RMA_SYNC_LEVEL_NONE,
                                     scorep_mpi_win_handle( win ),
                                     scorep_mpi_epoch_get_group_handle( win, SCOREP_MPI_RMA_EXPOSURE_EPOCH ) );
            }

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_TEST ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_TEST ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_UNLOCK ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_unlock
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_unlock( int     rank,
                MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_UNLOCK ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_UNLOCK ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_unlock( rank, win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_RmaWindowHandle win_handle = scorep_mpi_win_handle( win );

            scorep_mpi_rma_request_foreach_to_target( scorep_mpi_win_handle( win ),
                                                      rank,
                                                      scorep_mpi_rma_request_write_full_completion );

            SCOREP_RmaReleaseLock( scorep_mpi_win_handle( win ),
                                   rank, 0 );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_UNLOCK ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_UNLOCK ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_UNLOCK_ALL ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( MPI_Win_unlock_all )
/**
 * Measurement wrapper for MPI_Win_unlock_all
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma
 *
 * Triggers an enter event, a release-lock event, and an exit event.
 *
 */
int
MPI_Win_unlock_all( MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_UNLOCK_ALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_UNLOCK_ALL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_unlock_all( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_RmaWindowHandle win_handle = scorep_mpi_win_handle( win );

            scorep_mpi_rma_request_foreach_on_window( scorep_mpi_win_handle( win ),
                                                      scorep_mpi_rma_request_write_full_completion );

            SCOREP_RmaReleaseLock( scorep_mpi_win_handle( win ),
                                   SCOREP_ALL_TARGET_RANKS, 0 );

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_UNLOCK_ALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_UNLOCK_ALL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_WAIT ) && !defined( SCOREP_MPI_NO_RMA )
/**
 * Measurement wrapper for MPI_Win_wait
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter and exit event.
 */
int
MPI_Win_wait( MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_WAIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_WAIT ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_wait( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            scorep_mpi_rma_request_foreach_on_window( scorep_mpi_win_handle( win ),
                                                      scorep_mpi_rma_request_write_standard_completion_and_remove );

            SCOREP_RmaGroupSync( SCOREP_RMA_SYNC_LEVEL_MEMORY |
                                 SCOREP_RMA_SYNC_LEVEL_PROCESS,
                                 scorep_mpi_win_handle( win ),
                                 scorep_mpi_epoch_get_group_handle( win, SCOREP_MPI_RMA_EXPOSURE_EPOCH ) );

            scorep_mpi_epoch_end( win, SCOREP_MPI_RMA_EXPOSURE_EPOCH );

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_WAIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_WAIT ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_GET_GROUP ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_get_group )
/**
 * Measurement wrapper for MPI_Win_get_group
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Group.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma
 * Triggers an enter event and an exit event. Furthermore, a communicator definition
 * event is called between enter and exit.
 */
int
MPI_Win_get_group( MPI_Win win, MPI_Group* group )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_GROUP ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_GROUP ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_get_group( win, group );
    SCOREP_EXIT_WRAPPED_REGION();
    if ( *group != MPI_GROUP_NULL )
    {
        scorep_mpi_group_create( *group );
    }

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_GROUP ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_GROUP ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 * @name C wrappers for error handling functions
 * @{
 */
#if HAVE( DECL_PMPI_WIN_CALL_ERRHANDLER ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_ERR ) && !defined( MPI_Win_call_errhandler )
/**
 * Measurement wrapper for MPI_Win_call_errhandler
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_err
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_call_errhandler call with enter and exit events.
 */
int
MPI_Win_call_errhandler( MPI_Win win, int errorcode )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_ERR );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CALL_ERRHANDLER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CALL_ERRHANDLER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_call_errhandler( win, errorcode );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CALL_ERRHANDLER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CALL_ERRHANDLER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_CREATE_ERRHANDLER ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_ERR ) && !defined( MPI_Win_create_errhandler )
/**
 * Measurement wrapper for MPI_Win_create_errhandler
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_err
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_create_errhandler call with enter and exit events.
 */
int
MPI_Win_create_errhandler( MPI_Win_errhandler_fn* function, MPI_Errhandler* errhandler )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_ERR );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_ERRHANDLER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_ERRHANDLER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_create_errhandler( function, errhandler );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_ERRHANDLER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_ERRHANDLER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_GET_ERRHANDLER ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_ERR ) && !defined( MPI_Win_get_errhandler )
/**
 * Measurement wrapper for MPI_Win_get_errhandler
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_err
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_get_errhandler call with enter and exit events.
 */
int
MPI_Win_get_errhandler( MPI_Win win, MPI_Errhandler* errhandler )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_ERR );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_ERRHANDLER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_ERRHANDLER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_get_errhandler( win, errhandler );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_ERRHANDLER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_ERRHANDLER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_SET_ERRHANDLER ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_ERR ) && !defined( MPI_Win_set_errhandler )
/**
 * Measurement wrapper for MPI_Win_set_errhandler
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_err
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_set_errhandler call with enter and exit events.
 */
int
MPI_Win_set_errhandler( MPI_Win win, MPI_Errhandler errhandler )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_ERR );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_ERRHANDLER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_ERRHANDLER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_set_errhandler( win, errhandler );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_ERRHANDLER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_ERRHANDLER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 * @name C wrappers for external interface functions
 * @{
 */
#if HAVE( DECL_PMPI_WIN_CREATE_KEYVAL ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXT ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_create_keyval )
/**
 * Measurement wrapper for MPI_Win_create_keyval
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_create_keyval call with enter and exit events.
 */
int
MPI_Win_create_keyval( MPI_Win_copy_attr_function* win_copy_attr_fn, MPI_Win_delete_attr_function* win_delete_attr_fn, int* win_keyval, void* extra_state )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_KEYVAL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_KEYVAL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_create_keyval( win_copy_attr_fn, win_delete_attr_fn, win_keyval, extra_state );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_KEYVAL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_CREATE_KEYVAL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_DELETE_ATTR ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXT ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_delete_attr )
/**
 * Measurement wrapper for MPI_Win_delete_attr
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_delete_attr call with enter and exit events.
 */
int
MPI_Win_delete_attr( MPI_Win win, int win_keyval )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_DELETE_ATTR ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_DELETE_ATTR ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_delete_attr( win, win_keyval );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_DELETE_ATTR ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_DELETE_ATTR ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_FREE_KEYVAL ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXT ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_free_keyval )
/**
 * Measurement wrapper for MPI_Win_free_keyval
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_free_keyval call with enter and exit events.
 */
int
MPI_Win_free_keyval( int* win_keyval )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FREE_KEYVAL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FREE_KEYVAL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_free_keyval( win_keyval );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FREE_KEYVAL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_FREE_KEYVAL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_GET_ATTR ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXT ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_get_attr )
/**
 * Measurement wrapper for MPI_Win_get_attr
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_get_attr call with enter and exit events.
 */
int
MPI_Win_get_attr( MPI_Win win, int win_keyval, void* attribute_val, int* flag )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_ATTR ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_ATTR ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_get_attr( win, win_keyval, attribute_val, flag );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_ATTR ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_ATTR ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_GET_INFO ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_get_info )
/**
 * Measurement wrapper for MPI_Win_get_info
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_get_info call with enter and exit events.
 */
int
MPI_Win_get_info( MPI_Win win, MPI_Info* info_used )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_INFO ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_INFO ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_get_info( win, info_used );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_INFO ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_INFO ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_GET_NAME ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_get_name )
/**
 * Measurement wrapper for MPI_Win_get_name
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_get_name call with enter and exit events.
 */
int
MPI_Win_get_name( MPI_Win win, char* win_name, int* resultlen )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_NAME ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_NAME ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_get_name( win, win_name, resultlen );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_NAME ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_GET_NAME ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_SET_ATTR ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_EXT ) && !defined( MPI_Win_set_attr )
/**
 * Measurement wrapper for MPI_Win_set_attr
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_set_attr call with enter and exit events.
 */
int
MPI_Win_set_attr( MPI_Win win, int win_keyval, void* attribute_val )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_ATTR ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_ATTR ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_set_attr( win, win_keyval, attribute_val );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_ATTR ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_ATTR ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_SET_INFO ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_set_info )
/**
 * Measurement wrapper for MPI_Win_set_info
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_set_info call with enter and exit events.
 */
int
MPI_Win_set_info( MPI_Win win, MPI_Info info )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_INFO ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_INFO ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_set_info( win, info );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_INFO ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_INFO ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_WIN_SET_NAME ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_EXT ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_set_name )
/**
 * Measurement wrapper for MPI_Win_set_name
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_ext
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_set_name call with enter and exit events.
 */
int
MPI_Win_set_name( MPI_Win win, SCOREP_MPI_CONST_DECL char* win_name )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_EXT );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_NAME ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_NAME ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_set_name( win, win_name );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( return_val == MPI_SUCCESS )
    {
        scorep_mpi_win_set_name( win, win_name );
    }

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_NAME ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_SET_NAME ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 * @name C wrappers for handle conversion functions
 * @{
 */
#if HAVE( DECL_PMPI_WIN_C2F ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_MISC ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_c2f )
/**
 * Measurement wrapper for MPI_Win_c2f
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_misc
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_c2f call with enter and exit events.
 */
MPI_Fint
MPI_Win_c2f( MPI_Win win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_MISC );
    MPI_Fint  return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_C2F ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_C2F ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_c2f( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_C2F ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_C2F ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_WIN_F2C ) && !defined( SCOREP_MPI_NO_RMA ) && !defined( SCOREP_MPI_NO_MISC ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( MPI_Win_f2c )
/**
 * Measurement wrapper for MPI_Win_f2c
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup rma_misc
 * Triggers an enter and exit event.
 * It wraps the MPI_Win_f2c call with enter and exit events.
 */
MPI_Win
MPI_Win_f2c( MPI_Fint win )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_RMA_MISC );
    MPI_Win   return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_F2C ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_F2C ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Win_f2c( win );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_F2C ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_WIN_F2C ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 */
