/*
 * This file is part of the Score-P software (http://www.score-p.org)
 *
 * Copyright (c) 2009-2013,
 * RWTH Aachen University, Germany
 *
 * Copyright (c) 2009-2013,
 * Gesellschaft fuer numerische Simulation mbH Braunschweig, Germany
 *
 * Copyright (c) 2009-2018,
 * Technische Universitaet Dresden, Germany
 *
 * Copyright (c) 2009-2013,
 * University of Oregon, Eugene, USA
 *
 * Copyright (c) 2009-2019,
 * Forschungszentrum Juelich GmbH, Germany
 *
 * Copyright (c) 2009-2014,
 * German Research School for Simulation Sciences GmbH, Juelich/Aachen, Germany
 *
 * Copyright (c) 2009-2013,
 * Technische Universitaet Muenchen, Germany
 *
 * Copyright (c) 2016,
 * Technische Universitaet Darmstadt, Germany
 *
 * This software may be modified and distributed under the terms of
 * a BSD-style license.  See the COPYING file in the package base
 * directory for details.
 *
 */

/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2011                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2010-2011                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2003-2008                                                **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/


/**
 * @file
 * @ingroup    MPI_Wrapper
 *
 * @brief C interface wrappers for point-to-point communication
 */

#include <config.h>
#include "SCOREP_Mpi.h"
#include "scorep_mpi_communicator.h"
#include "scorep_mpi_request_mgmt.h"
#include <UTILS_Error.h>
#include <SCOREP_RuntimeManagement.h>
#include <SCOREP_InMeasurement.h>
#include <SCOREP_Events.h>

/**
 * @name Blocking
 * @{
 */
#if HAVE( DECL_PMPI_BSEND ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Bsend )
/**
 * Measurement wrapper for MPI_Bsend
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSend.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Bsend'
 * @li MPI send event
 * @li exit region 'MPI_Bsend'
 */
int
MPI_Bsend( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

#if !defined( SCOREP_MPI_NO_HOOKS )
    uint64_t start_time_stamp;
#endif

    int sz;
    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BSEND ] );

  #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
   #endif

            if ( dest != MPI_PROC_NULL )
            {
                PMPI_Type_size( datatype, &sz );
                SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                tag, ( uint64_t )count * sz );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BSEND ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Bsend( buf, count, datatype, dest, tag, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Bsend( buf, count, datatype, dest, tag, comm, start_time_stamp, return_val );
            }
#endif

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BSEND ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BSEND ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_RSEND ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Rsend )
/**
 * Measurement wrapper for MPI_Rsend
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSend.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Rsend'
 * @li MPI send event
 * @li exit region 'MPI_Rsend'
 */
int
MPI_Rsend( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

#if !defined( SCOREP_MPI_NO_HOOKS )
    uint64_t start_time_stamp;
#endif

    int sz;
    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RSEND ] );

  #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
   #endif

            if ( dest != MPI_PROC_NULL )
            {
                PMPI_Type_size( datatype, &sz );
                SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                tag, ( uint64_t )count * sz );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RSEND ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Rsend( buf, count, datatype, dest, tag, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Rsend( buf, count, datatype, dest, tag, comm, start_time_stamp, return_val );
            }
#endif

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RSEND ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RSEND ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_SEND ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Send )
/**
 * Measurement wrapper for MPI_Send
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSend.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Send'
 * @li MPI send event
 * @li exit region 'MPI_Send'
 */
int
MPI_Send( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

#if !defined( SCOREP_MPI_NO_HOOKS )
    uint64_t start_time_stamp;
#endif

    int sz;
    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SEND ] );

  #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
   #endif

            if ( dest != MPI_PROC_NULL )
            {
                PMPI_Type_size( datatype, &sz );
                SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                tag, ( uint64_t )count * sz );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SEND ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Send( buf, count, datatype, dest, tag, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Send( buf, count, datatype, dest, tag, comm, start_time_stamp, return_val );
            }
#endif

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SEND ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SEND ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_SSEND ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Ssend )
/**
 * Measurement wrapper for MPI_Ssend
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSend.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Ssend'
 * @li MPI send event
 * @li exit region 'MPI_Ssend'
 */
int
MPI_Ssend( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

#if !defined( SCOREP_MPI_NO_HOOKS )
    uint64_t start_time_stamp;
#endif

    int sz;
    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SSEND ] );

  #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
   #endif

            if ( dest != MPI_PROC_NULL )
            {
                PMPI_Type_size( datatype, &sz );
                SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                tag, ( uint64_t )count * sz );
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SSEND ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ssend( buf, count, datatype, dest, tag, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Ssend( buf, count, datatype, dest, tag, comm, start_time_stamp, return_val );
            }
#endif

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SSEND ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SSEND ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_RECV ) && !defined( SCOREP_MPI_NO_P2P )
/**
 * Measurement wrapper for MPI_Recv
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Recv'
 * @li MPI recv event
 * @li exit region 'MPI_Recv'
 */
int
MPI_Recv( void* buf,
          int count,
          MPI_Datatype datatype,
          int source, int tag,
          MPI_Comm comm,
          MPI_Status* status )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int  event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int  event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int        return_val;
    int        sz;
    uint64_t   start_time_stamp;
    MPI_Status mystatus;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RECV ] );

#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
#endif

            if ( status == MPI_STATUS_IGNORE )
            {
                status = &mystatus;
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RECV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Recv( buf, count, datatype, source, tag, comm, status );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Recv( buf, count, datatype, source, tag, comm, status, start_time_stamp, return_val );
            }
#endif

            if ( source != MPI_PROC_NULL && return_val == MPI_SUCCESS )
            {
                PMPI_Type_size( datatype, &sz );
                PMPI_Get_count( status, datatype, &count );
                SCOREP_MpiRecv( status->MPI_SOURCE, SCOREP_MPI_COMM_HANDLE( comm ),
                                status->MPI_TAG, ( uint64_t )count * sz );
            }

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RECV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RECV ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_PROBE ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Probe )
/**
 * Measurement wrapper for MPI_Probe
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Triggers an enter and exit event.
 * It wraps the MPI_Probe call with enter and exit events.
 */
int
MPI_Probe( int source, int tag, MPI_Comm comm, MPI_Status* status )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_PROBE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_PROBE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Probe( source, tag, comm, status );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_PROBE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_PROBE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_MPROBE ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Mprobe )
/**
 * Measurement wrapper for MPI_Mprobe
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup p2p
 * Triggers an enter and exit event.
 * It wraps the MPI_Mprobe call with enter and exit events.
 */
int
MPI_Mprobe( int source, int tag, MPI_Comm comm, MPI_Message* message, MPI_Status* status )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_MPROBE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_MPROBE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Mprobe( source, tag, comm, message, status );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_MPROBE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_MPROBE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_IMPROBE ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Improbe )
/**
 * Measurement wrapper for MPI_Improbe
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup p2p
 * Triggers an enter and exit event.
 * It wraps the MPI_Improbe call with enter and exit events.
 */
int
MPI_Improbe( int source, int tag, MPI_Comm comm, int* flag, MPI_Message* message, MPI_Status* status )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IMPROBE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IMPROBE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Improbe( source, tag, comm, flag, message, status );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IMPROBE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IMPROBE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_MRECV ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Mrecv )
/**
 * Measurement wrapper for MPI_Mrecv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup p2p
 * Triggers an enter and exit event.
 * It wraps the MPI_Mrecv call with enter and exit events.
 */
int
MPI_Mrecv( void* buf, int count, MPI_Datatype datatype, MPI_Message* message, MPI_Status* status )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_MRECV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_MRECV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Mrecv( buf, count, datatype, message, status );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_MRECV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_MRECV ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_IMRECV ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Imrecv )
/**
 * Measurement wrapper for MPI_Imrecv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup p2p
 * Triggers an enter and exit event.
 * It wraps the MPI_Imrecv call with enter and exit events.
 */
int
MPI_Imrecv( void* buf, int count, MPI_Datatype datatype, MPI_Message* message, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IMRECV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IMRECV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Imrecv( buf, count, datatype, message, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IMRECV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IMRECV ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif


#if HAVE( DECL_PMPI_SENDRECV ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Sendrecv )
/**
 * Measurement wrapper for MPI_Sendrecv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSendrecv.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * @li enter region 'MPI_Sendrecv'
 * @li MPI send event
 * @li MPI receive event
 * @li exit region 'MPI_Sendrecv'
 */
int
MPI_Sendrecv( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, int dest, int sendtag, void* recvbuf, int recvcount, MPI_Datatype recvtype, int source, int recvtag, MPI_Comm comm, MPI_Status* status )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int  event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int  event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int        return_val;
    MPI_Status mystatus;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SENDRECV ] );

            if ( dest != MPI_PROC_NULL )
            {
                int sz;
                PMPI_Type_size( sendtype, &sz );
                SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                sendtag, ( uint64_t )sendcount * sz );
            }
            if ( status == MPI_STATUS_IGNORE )
            {
                status = &mystatus;
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SENDRECV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Sendrecv( sendbuf, sendcount, sendtype, dest, sendtag, recvbuf, recvcount, recvtype, source, recvtag, comm, status );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( source != MPI_PROC_NULL && return_val == MPI_SUCCESS )
            {
                int sz;
                PMPI_Type_size( recvtype, &sz );
                PMPI_Get_count( status, recvtype, &recvcount );
                SCOREP_MpiRecv( status->MPI_SOURCE, SCOREP_MPI_COMM_HANDLE( comm ),
                                status->MPI_TAG, ( uint64_t )recvcount * sz );
            }

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SENDRECV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SENDRECV ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_SENDRECV_REPLACE ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Sendrecv_replace )
/**
 * Measurement wrapper for MPI_Sendrecv_replace
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSendrecv.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * @li enter region 'MPI_Sendrecv_replace'
 * @li MPI send event
 * @li MPI receive event
 * @li exit region 'MPI_Sendrecv_replace'
 */
int
MPI_Sendrecv_replace( void* buf, int count, MPI_Datatype datatype, int dest, int sendtag, int source, int recvtag, MPI_Comm comm, MPI_Status* status )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int    event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int    event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int          return_val;
    MPI_Status   mystatus;
    int          sendcount = count;
    int          recvcount = count;
    MPI_Datatype sendtype  = datatype;
    MPI_Datatype recvtype  = datatype;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SENDRECV_REPLACE ] );

            if ( dest != MPI_PROC_NULL )
            {
                int sz;
                PMPI_Type_size( sendtype, &sz );
                SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                sendtag, ( uint64_t )sendcount * sz );
            }
            if ( status == MPI_STATUS_IGNORE )
            {
                status = &mystatus;
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SENDRECV_REPLACE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Sendrecv_replace( buf, count, datatype, dest, sendtag, source, recvtag, comm, status );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( source != MPI_PROC_NULL && return_val == MPI_SUCCESS )
            {
                int sz;
                PMPI_Type_size( recvtype, &sz );
                PMPI_Get_count( status, recvtype, &recvcount );
                SCOREP_MpiRecv( status->MPI_SOURCE, SCOREP_MPI_COMM_HANDLE( comm ),
                                status->MPI_TAG, ( uint64_t )recvcount * sz );
            }

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SENDRECV_REPLACE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SENDRECV_REPLACE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 * @name Non-blocking
 * @{
 */

#if HAVE( DECL_PMPI_IBSEND ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Ibsend )
/**
 * Measurement wrapper for MPI_Ibsend
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpIsend.w
 * @note C interface
 * @note Introduced with MPI 1
 * @ingroup p2p
 */
int
MPI_Ibsend( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int           event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int           event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    const int           xnb_active                 = ( scorep_mpi_enabled & SCOREP_MPI_ENABLED_XNONBLOCK );
    int                 return_val;
    SCOREP_MpiRequestId reqid;

#if !defined( SCOREP_MPI_NO_HOOKS )
    uint64_t start_time_stamp;
#endif

    int sz;
    if ( event_gen_active )
    {
        reqid = scorep_mpi_get_request_id();
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBSEND ] );

#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
#endif

            if ( dest != MPI_PROC_NULL )
            {
                PMPI_Type_size( datatype, &sz );
                if ( xnb_active )
                {
                    SCOREP_MpiIsend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                     tag, ( uint64_t )count * sz, reqid );
                }
                else
                {
                    SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                    tag, ( uint64_t )count * sz );
                }
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBSEND ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ibsend( buf, count, datatype, dest, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( xnb_active && dest != MPI_PROC_NULL && return_val == MPI_SUCCESS )
            {
                scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_SEND, SCOREP_MPI_REQUEST_FLAG_NONE,
                                               tag, dest, ( uint64_t )count * sz, datatype, comm, reqid );

#if !defined( SCOREP_MPI_NO_HOOKS )
                if ( SCOREP_IS_MPI_HOOKS_ON )
                {
                    SCOREP_Hooks_Post_MPI_Ibsend( buf, count, datatype, dest, tag, comm, request, start_time_stamp, return_val );
                }
#endif
            }

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBSEND ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBSEND ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IRSEND ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Irsend )
/**
 * Measurement wrapper for MPI_Irsend
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpIsend.w
 * @note C interface
 * @note Introduced with MPI 1
 * @ingroup p2p
 */
int
MPI_Irsend( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int           event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int           event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    const int           xnb_active                 = ( scorep_mpi_enabled & SCOREP_MPI_ENABLED_XNONBLOCK );
    int                 return_val;
    SCOREP_MpiRequestId reqid;

#if !defined( SCOREP_MPI_NO_HOOKS )
    uint64_t start_time_stamp;
#endif

    int sz;
    if ( event_gen_active )
    {
        reqid = scorep_mpi_get_request_id();
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IRSEND ] );

#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
#endif

            if ( dest != MPI_PROC_NULL )
            {
                PMPI_Type_size( datatype, &sz );
                if ( xnb_active )
                {
                    SCOREP_MpiIsend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                     tag, ( uint64_t )count * sz, reqid );
                }
                else
                {
                    SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                    tag, ( uint64_t )count * sz );
                }
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IRSEND ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Irsend( buf, count, datatype, dest, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( xnb_active && dest != MPI_PROC_NULL && return_val == MPI_SUCCESS )
            {
                scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_SEND, SCOREP_MPI_REQUEST_FLAG_NONE,
                                               tag, dest, ( uint64_t )count * sz, datatype, comm, reqid );

#if !defined( SCOREP_MPI_NO_HOOKS )
                if ( SCOREP_IS_MPI_HOOKS_ON )
                {
                    SCOREP_Hooks_Post_MPI_Irsend( buf, count, datatype, dest, tag, comm, request, start_time_stamp, return_val );
                }
#endif
            }

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IRSEND ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IRSEND ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ISEND ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Isend )
/**
 * Measurement wrapper for MPI_Isend
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpIsend.w
 * @note C interface
 * @note Introduced with MPI 1
 * @ingroup p2p
 */
int
MPI_Isend( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int           event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int           event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    const int           xnb_active                 = ( scorep_mpi_enabled & SCOREP_MPI_ENABLED_XNONBLOCK );
    int                 return_val;
    SCOREP_MpiRequestId reqid;

#if !defined( SCOREP_MPI_NO_HOOKS )
    uint64_t start_time_stamp;
#endif

    int sz;
    if ( event_gen_active )
    {
        reqid = scorep_mpi_get_request_id();
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISEND ] );

#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
#endif

            if ( dest != MPI_PROC_NULL )
            {
                PMPI_Type_size( datatype, &sz );
                if ( xnb_active )
                {
                    SCOREP_MpiIsend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                     tag, ( uint64_t )count * sz, reqid );
                }
                else
                {
                    SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                    tag, ( uint64_t )count * sz );
                }
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISEND ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Isend( buf, count, datatype, dest, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( xnb_active && dest != MPI_PROC_NULL && return_val == MPI_SUCCESS )
            {
                scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_SEND, SCOREP_MPI_REQUEST_FLAG_NONE,
                                               tag, dest, ( uint64_t )count * sz, datatype, comm, reqid );

#if !defined( SCOREP_MPI_NO_HOOKS )
                if ( SCOREP_IS_MPI_HOOKS_ON )
                {
                    SCOREP_Hooks_Post_MPI_Isend( buf, count, datatype, dest, tag, comm, request, start_time_stamp, return_val );
                }
#endif
            }

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISEND ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISEND ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ISSEND ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Issend )
/**
 * Measurement wrapper for MPI_Issend
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpIsend.w
 * @note C interface
 * @note Introduced with MPI 1
 * @ingroup p2p
 */
int
MPI_Issend( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int           event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int           event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    const int           xnb_active                 = ( scorep_mpi_enabled & SCOREP_MPI_ENABLED_XNONBLOCK );
    int                 return_val;
    SCOREP_MpiRequestId reqid;

#if !defined( SCOREP_MPI_NO_HOOKS )
    uint64_t start_time_stamp;
#endif

    int sz;
    if ( event_gen_active )
    {
        reqid = scorep_mpi_get_request_id();
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISSEND ] );

#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
#endif

            if ( dest != MPI_PROC_NULL )
            {
                PMPI_Type_size( datatype, &sz );
                if ( xnb_active )
                {
                    SCOREP_MpiIsend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                     tag, ( uint64_t )count * sz, reqid );
                }
                else
                {
                    SCOREP_MpiSend( dest, SCOREP_MPI_COMM_HANDLE( comm ),
                                    tag, ( uint64_t )count * sz );
                }
            }
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISSEND ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Issend( buf, count, datatype, dest, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            if ( xnb_active && dest != MPI_PROC_NULL && return_val == MPI_SUCCESS )
            {
                scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_SEND, SCOREP_MPI_REQUEST_FLAG_NONE,
                                               tag, dest, ( uint64_t )count * sz, datatype, comm, reqid );

#if !defined( SCOREP_MPI_NO_HOOKS )
                if ( SCOREP_IS_MPI_HOOKS_ON )
                {
                    SCOREP_Hooks_Post_MPI_Issend( buf, count, datatype, dest, tag, comm, request, start_time_stamp, return_val );
                }
#endif
            }

            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISSEND ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISSEND ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_IRECV ) && !defined( SCOREP_MPI_NO_P2P )
/**
 * Measurement wrapper for MPI_Irecv
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Triggers an enter and exit event.
 */
int
MPI_Irecv( void*        buf,
           int          count,
           MPI_Datatype datatype,
           int          source,
           int          tag,
           MPI_Comm     comm,
           MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    const int xnb_active                 = ( scorep_mpi_enabled & SCOREP_MPI_ENABLED_XNONBLOCK );
    int       return_val;
    uint64_t  start_time_stamp;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IRECV ] );

#if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                start_time_stamp = SCOREP_GetLastTimeStamp();
            }
#endif
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IRECV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Irecv( buf, count, datatype, source, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( source != MPI_PROC_NULL && return_val == MPI_SUCCESS )
    {
        SCOREP_MpiRequestId reqid = scorep_mpi_get_request_id();
        int                 sz;
        PMPI_Type_size( datatype, &sz );

        if ( event_gen_active_for_group && xnb_active )
        {
            SCOREP_MpiIrecvRequest( reqid );
        }

        scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_RECV, SCOREP_MPI_REQUEST_FLAG_NONE,
                                       tag, 0, ( uint64_t )count * sz, datatype, comm, reqid );
#if !defined( SCOREP_MPI_NO_HOOKS )
        if ( event_gen_active_for_group && SCOREP_IS_MPI_HOOKS_ON )
        {
            SCOREP_Hooks_Post_MPI_Irecv( buf, count, datatype, source, tag, comm, request, start_time_stamp, return_val );
        }
#endif
    }

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IRECV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IRECV ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_IPROBE ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Iprobe )
/**
 * Measurement wrapper for MPI_Iprobe
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Triggers an enter and exit event.
 * It wraps the MPI_Iprobe call with enter and exit events.
 */
int
MPI_Iprobe( int source, int tag, MPI_Comm comm, int* flag, MPI_Status* status )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IPROBE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IPROBE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Iprobe( source, tag, comm, flag, status );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IPROBE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IPROBE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 * @name Persistent requests
 * @{
 */

#if HAVE( DECL_PMPI_BSEND_INIT ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Bsend_init )
/**
 * Measurement wrapper for MPI_Bsend_init
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSendinit.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Bsend_init'
 * @li exit region 'MPI_Bsend_init'
 */
int
MPI_Bsend_init( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;
    int       sz;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BSEND_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BSEND_INIT ] );
        }
    }

    PMPI_Type_size( datatype, &sz );

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Bsend_init( buf, count, datatype, dest, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( dest != MPI_PROC_NULL && return_val == MPI_SUCCESS )
    {
        scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_SEND, SCOREP_MPI_REQUEST_FLAG_IS_PERSISTENT,
                                       tag, dest, ( uint64_t )count * sz, datatype, comm,
                                       scorep_mpi_get_request_id() );
    }
    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BSEND_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BSEND_INIT ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_RSEND_INIT ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Rsend_init )
/**
 * Measurement wrapper for MPI_Rsend_init
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSendinit.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Rsend_init'
 * @li exit region 'MPI_Rsend_init'
 */
int
MPI_Rsend_init( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;
    int       sz;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RSEND_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RSEND_INIT ] );
        }
    }

    PMPI_Type_size( datatype, &sz );

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Rsend_init( buf, count, datatype, dest, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( dest != MPI_PROC_NULL && return_val == MPI_SUCCESS )
    {
        scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_SEND, SCOREP_MPI_REQUEST_FLAG_IS_PERSISTENT,
                                       tag, dest, ( uint64_t )count * sz, datatype, comm,
                                       scorep_mpi_get_request_id() );
    }
    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RSEND_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RSEND_INIT ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_SEND_INIT ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Send_init )
/**
 * Measurement wrapper for MPI_Send_init
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSendinit.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Send_init'
 * @li exit region 'MPI_Send_init'
 */
int
MPI_Send_init( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;
    int       sz;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SEND_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SEND_INIT ] );
        }
    }

    PMPI_Type_size( datatype, &sz );

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Send_init( buf, count, datatype, dest, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( dest != MPI_PROC_NULL && return_val == MPI_SUCCESS )
    {
        scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_SEND, SCOREP_MPI_REQUEST_FLAG_IS_PERSISTENT,
                                       tag, dest, ( uint64_t )count * sz, datatype, comm,
                                       scorep_mpi_get_request_id() );
    }
    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SEND_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SEND_INIT ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_SSEND_INIT ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Ssend_init )
/**
 * Measurement wrapper for MPI_Ssend_init
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_PtpSendinit.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Sequence of events:
 * @li enter region 'MPI_Ssend_init'
 * @li exit region 'MPI_Ssend_init'
 */
int
MPI_Ssend_init( SCOREP_MPI_CONST_DECL void* buf, int count, MPI_Datatype datatype, int dest, int tag, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;
    int       sz;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SSEND_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SSEND_INIT ] );
        }
    }

    PMPI_Type_size( datatype, &sz );

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ssend_init( buf, count, datatype, dest, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( dest != MPI_PROC_NULL && return_val == MPI_SUCCESS )
    {
        scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_SEND, SCOREP_MPI_REQUEST_FLAG_IS_PERSISTENT,
                                       tag, dest, ( uint64_t )count * sz, datatype, comm,
                                       scorep_mpi_get_request_id() );
    }
    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SSEND_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SSEND_INIT ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_RECV_INIT ) && !defined( SCOREP_MPI_NO_P2P )
/**
 * Measurement wrapper for MPI_Recv_init
 * @note Manually adapted wrapper
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Triggers an enter and exit event.
 */
int
MPI_Recv_init( void*        buf,
               int          count,
               MPI_Datatype datatype,
               int          source,
               int          tag,
               MPI_Comm     comm,
               MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();

        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RECV_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RECV_INIT ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Recv_init( buf, count, datatype, source, tag, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();
    if ( source != MPI_PROC_NULL && return_val == MPI_SUCCESS )
    {
        int sz;
        PMPI_Type_size( datatype, &sz );
        scorep_mpi_request_p2p_create( *request, SCOREP_MPI_REQUEST_TYPE_RECV, SCOREP_MPI_REQUEST_FLAG_IS_PERSISTENT,
                                       tag, source, ( uint64_t )count * sz, datatype, comm,
                                       scorep_mpi_get_request_id() );

        #if !defined( SCOREP_MPI_NO_HOOKS )
        if ( SCOREP_IS_MPI_HOOKS_ON )
        {
            SCOREP_Hooks_Post_MPI_Recv_init( buf, count, datatype, source, tag, comm, request, 0, return_val );
        }
    #endif
    }

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RECV_INIT ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_RECV_INIT ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 * @name Auxiliary functions
 * @{
 */

#if HAVE( DECL_PMPI_BUFFER_ATTACH ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Buffer_attach )
/**
 * Measurement wrapper for MPI_Buffer_attach
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Triggers an enter and exit event.
 * It wraps the MPI_Buffer_attach call with enter and exit events.
 */
int
MPI_Buffer_attach( void* buffer, int size )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BUFFER_ATTACH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BUFFER_ATTACH ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Buffer_attach( buffer, size );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BUFFER_ATTACH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BUFFER_ATTACH ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_BUFFER_DETACH ) && !defined( SCOREP_MPI_NO_EXTRA ) && !defined( SCOREP_MPI_NO_P2P ) && !defined( MPI_Buffer_detach )
/**
 * Measurement wrapper for MPI_Buffer_detach
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup p2p
 * Triggers an enter and exit event.
 * It wraps the MPI_Buffer_detach call with enter and exit events.
 */
int
MPI_Buffer_detach( void* buffer, int* size )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_P2P );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BUFFER_DETACH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BUFFER_DETACH ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Buffer_detach( buffer, size );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BUFFER_DETACH ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BUFFER_DETACH ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

/**
 * @}
 */
