/*
 * This file is part of the Score-P software (http://www.score-p.org)
 *
 * Copyright (c) 2009-2013,
 * RWTH Aachen University, Germany
 *
 * Copyright (c) 2009-2013,
 * Gesellschaft fuer numerische Simulation mbH Braunschweig, Germany
 *
 * Copyright (c) 2009-2017,
 * Technische Universitaet Dresden, Germany
 *
 * Copyright (c) 2009-2013,
 * University of Oregon, Eugene, USA
 *
 * Copyright (c) 2009-2015, 2019,
 * Forschungszentrum Juelich GmbH, Germany
 *
 * Copyright (c) 2009-2014,
 * German Research School for Simulation Sciences GmbH, Juelich/Aachen, Germany
 *
 * Copyright (c) 2009-2013,
 * Technische Universitaet Muenchen, Germany
 *
 * Copyright (c) 2016,
 * Technische Universitaet Darmstadt, Germany
 *
 * This software may be modified and distributed under the terms of
 * a BSD-style license.  See the COPYING file in the package base
 * directory for details.
 *
 */

/****************************************************************************
**  SCALASCA    http://www.scalasca.org/                                   **
*****************************************************************************
**  Copyright (c) 1998-2011                                                **
**  Forschungszentrum Juelich GmbH, Juelich Supercomputing Centre          **
**                                                                         **
**  Copyright (c) 2010-2011                                                **
**  German Research School for Simulation Sciences GmbH,                   **
**  Laboratory for Parallel Programming                                    **
**                                                                         **
**  Copyright (c) 2003-2008                                                **
**  University of Tennessee, Innovative Computing Laboratory               **
**                                                                         **
**  See the file COPYRIGHT in the package base directory for details       **
****************************************************************************/


/**
 * @file
 * @ingroup    MPI_Wrapper
 *
 * @brief C interface wrappers for collective communication
 */

#include <config.h>
#include "SCOREP_Mpi.h"
#include "scorep_mpi_communicator.h"
#include <SCOREP_RuntimeManagement.h>
#include <SCOREP_InMeasurement.h>
#include <SCOREP_Events.h>

/**
 * @name C wrappers
 * @{
 */

#if HAVE( DECL_PMPI_ALLGATHER )
/**
 * Measurement wrapper for MPI_Allgather
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Allgather( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            recvsz, sendsz, N;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Comm_size( comm, &N );
            PMPI_Type_size( recvtype, &recvsz );

     #if HAVE( MPI_IN_PLACE )
            if ( sendbuf != MPI_IN_PLACE )
     #endif
            {
                PMPI_Type_size( sendtype, &sendsz );
                sendbytes = ( uint64_t )N * sendcount * sendsz;
                recvbytes = ( uint64_t )N * recvcount * recvsz;
            }
     #if HAVE( MPI_IN_PLACE )
            else
            {
                sendbytes = recvbytes = ( uint64_t )( N - 1 ) * recvcount * recvsz;
            }
     #endif


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLGATHER ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLGATHER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Allgather( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Allgather( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_ALLGATHER,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLGATHER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLGATHER ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ALLGATHERV ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Allgatherv )
/**
 * Measurement wrapper for MPI_Allgatherv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Allgatherv( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, SCOREP_MPI_CONST_DECL int* recvcounts, SCOREP_MPI_CONST_DECL int* displs, MPI_Datatype recvtype, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int32_t        recvcount, recvsz, sendsz, i, N, me;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Comm_size( comm, &N );
            PMPI_Type_size( recvtype, &recvsz );
            PMPI_Comm_rank( comm, &me );

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf != MPI_IN_PLACE )
      #endif
            {
                PMPI_Type_size( sendtype, &sendsz );
                sendbytes = ( uint64_t )N * sendcount * sendsz;
            }
      #if HAVE( MPI_IN_PLACE )
            else
            {
                sendbytes = ( uint64_t )( N - 1 ) * recvcounts[ me ] * recvsz;
            }
      #endif

            recvcount = 0;
            for ( i = 0; i < N; i++ )
            {
                recvcount += recvcounts[ i ];
            }

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf == MPI_IN_PLACE )
            {
                recvcount -= recvcounts[ me ];
            }
      #endif
            recvbytes = ( uint64_t )recvcount * recvsz;


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLGATHERV ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLGATHERV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Allgatherv( sendbuf, sendcount, sendtype, recvbuf, recvcounts, displs, recvtype, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Allgatherv( sendbuf, sendcount, sendtype, recvbuf, recvcounts, displs, recvtype, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_ALLGATHERV,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLGATHERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLGATHERV ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ALLREDUCE ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Allreduce )
/**
 * Measurement wrapper for MPI_Allreduce
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Allreduce( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int count, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int32_t        sz, N;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Type_size( datatype, &sz );
            PMPI_Comm_size( comm, &N );

      #if HAVE( HAS_MPI_IN_PLACE )
            if ( sendbuf == MPI_IN_PLACE )
            {
                sendbytes = recvbytes = ( uint64_t )( N - 1 ) * count * sz;
            }
            else
      #endif
            {
                sendbytes = recvbytes = ( uint64_t )N * count * sz;
            }


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLREDUCE ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLREDUCE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Allreduce( sendbuf, recvbuf, count, datatype, op, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Allreduce( sendbuf, recvbuf, count, datatype, op, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_ALLREDUCE,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLREDUCE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLREDUCE ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ALLTOALL ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Alltoall )
/**
 * Measurement wrapper for MPI_Alltoall
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Alltoall( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int32_t        recvsz, N;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Type_size( recvtype, &recvsz );
            PMPI_Comm_size( comm, &N );

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf == MPI_IN_PLACE )
            {
                --N;
            }
      #endif

            sendbytes = ( uint64_t )N * recvcount * recvsz;
            recvbytes = sendbytes;


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALL ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Alltoall( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Alltoall( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_ALLTOALL,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALL ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ALLTOALLV ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Alltoallv )
/**
 * Measurement wrapper for MPI_Alltoallv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Alltoallv( SCOREP_MPI_CONST_DECL void* sendbuf, SCOREP_MPI_CONST_DECL int* sendcounts, SCOREP_MPI_CONST_DECL int* sdispls, MPI_Datatype sendtype, void* recvbuf, SCOREP_MPI_CONST_DECL int* recvcounts, SCOREP_MPI_CONST_DECL int* rdispls, MPI_Datatype recvtype, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            recvcount = 0, recvsz, sendsz, N, i, me;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Comm_size( comm, &N );
            PMPI_Type_size( recvtype, &recvsz );

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf != MPI_IN_PLACE )
      #endif
            {
                PMPI_Type_size( sendtype, &sendsz );
                for ( i = 0; i < N; i++ )
                {
                    recvbytes += ( uint64_t )recvcounts[ i ] * recvsz;
                    sendbytes += ( uint64_t )sendcounts[ i ] * sendsz;
                }
            }
      #if HAVE( MPI_IN_PLACE )
            else
            {
                PMPI_Comm_rank( comm, &me );
                for ( i = 0; i < N; i++ )
                {
                    recvcount += recvcounts[ i ];
                }

                recvcount -= recvcounts[ me ];

                sendbytes = recvbytes = ( uint64_t )recvcount * recvsz;
            }
      #endif


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALLV ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALLV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Alltoallv( sendbuf, sendcounts, sdispls, sendtype, recvbuf, recvcounts, rdispls, recvtype, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Alltoallv( sendbuf, sendcounts, sdispls, sendtype, recvbuf, recvcounts, rdispls, recvtype, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_ALLTOALLV,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALLV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALLV ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ALLTOALLW ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Alltoallw )
/**
 * Measurement wrapper for MPI_Alltoallw
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Alltoallw( SCOREP_MPI_CONST_DECL void* sendbuf, SCOREP_MPI_CONST_DECL int sendcounts[], SCOREP_MPI_CONST_DECL int sdispls[], SCOREP_MPI_CONST_DECL MPI_Datatype sendtypes[], void* recvbuf, SCOREP_MPI_CONST_DECL int recvcounts[], SCOREP_MPI_CONST_DECL int rdispls[], SCOREP_MPI_CONST_DECL MPI_Datatype recvtypes[], MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            recvsz, sendsz, N, i, me;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Comm_size( comm, &N );

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf != MPI_IN_PLACE )
      #endif
            {
                for ( i = 0; i < N; i++ )
                {
                    PMPI_Type_size( recvtypes[ i ], &recvsz );
                    recvbytes += ( uint64_t )recvcounts[ i ] * recvsz;

                    PMPI_Type_size( sendtypes[ i ], &sendsz );
                    sendbytes += ( uint64_t )sendcounts[ i ] * sendsz;
                }
            }
      #if HAVE( MPI_IN_PLACE )
            else
            {
                PMPI_Comm_rank( comm, &me );

                for ( i = 0; i < N; i++ )
                {
                    PMPI_Type_size( recvtypes[ i ], &recvsz );
                    recvbytes += ( uint64_t )recvcounts[ i ] * recvsz;
                }

                PMPI_Type_size( recvtypes[ me ], &recvsz );
                recvbytes -= ( uint64_t )recvcounts[ me ] * recvsz;

                sendbytes = recvbytes;
            }
      #endif


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALLW ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALLW ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Alltoallw( sendbuf, sendcounts, sdispls, sendtypes, recvbuf, recvcounts, rdispls, recvtypes, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Alltoallw( sendbuf, sendcounts, sdispls, sendtypes, recvbuf, recvcounts, rdispls, recvtypes, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_ALLTOALLW,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALLW ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ALLTOALLW ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_BARRIER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Barrier )
/**
 * Measurement wrapper for MPI_Barrier
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Barrier( MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int      event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int      event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int            return_val;
    uint64_t       start_time_stamp;
    SCOREP_MpiRank root_loc = SCOREP_INVALID_ROOT_RANK;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BARRIER ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BARRIER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Barrier( comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Barrier( comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_BARRIER,
                                     0,
                                     0 );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BARRIER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BARRIER ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_BCAST ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Bcast )
/**
 * Measurement wrapper for MPI_Bcast
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Bcast( void* buffer, int count, MPI_Datatype datatype, int root, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int32_t        sz, N, me;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = root;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Type_size( datatype, &sz );
            PMPI_Comm_rank( comm, &me );
            if ( me == root )
            {
                PMPI_Comm_size( comm, &N );
            }
            else
            {
                N = 0;
            }

            sendbytes = ( uint64_t )N * count * sz;
            recvbytes = ( uint64_t )count * sz;


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BCAST ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BCAST ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Bcast( buffer, count, datatype, root, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Bcast( buffer, count, datatype, root, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_BCAST,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BCAST ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_BCAST ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_EXSCAN ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Exscan )
/**
 * Measurement wrapper for MPI_Exscan
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Exscan( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int count, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int32_t        sz, me, N;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Type_size( datatype, &sz );
            PMPI_Comm_rank( comm, &me );
            PMPI_Comm_size( comm, &N );

            sendbytes = ( uint64_t )( N - me - 1 ) * sz * count;
            recvbytes = ( uint64_t )me * sz * count;


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_EXSCAN ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_EXSCAN ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Exscan( sendbuf, recvbuf, count, datatype, op, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Exscan( sendbuf, recvbuf, count, datatype, op, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_EXSCAN,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_EXSCAN ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_EXSCAN ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_GATHER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Gather )
/**
 * Measurement wrapper for MPI_Gather
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Gather( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, int root, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            sendsz, recvsz, N, me;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = root;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf != MPI_IN_PLACE )
      #endif
            {
                PMPI_Type_size( sendtype, &sendsz );
                sendbytes = ( uint64_t )sendcount * sendsz;
            }
            /* MPI_IN_PLACE: sendbytes is initialized to 0 */

            PMPI_Comm_rank( comm, &me );
            if ( me == root )
            {
                PMPI_Comm_size( comm, &N );
                PMPI_Type_size( recvtype, &recvsz );
        #if HAVE( MPI_IN_PLACE )
                if ( sendbuf == MPI_IN_PLACE )
                {
                    --N;
                }
        #endif
                recvbytes = ( uint64_t )N * recvcount * recvsz;
            }


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GATHER ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GATHER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Gather( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, root, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Gather( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, root, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_GATHER,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GATHER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GATHER ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_GATHERV ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Gatherv )
/**
 * Measurement wrapper for MPI_Gatherv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Gatherv( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, SCOREP_MPI_CONST_DECL int* recvcounts, SCOREP_MPI_CONST_DECL int* displs, MPI_Datatype recvtype, int root, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            recvsz, sendsz, me, N, i;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = root;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf != MPI_IN_PLACE )
      #endif
            {
                PMPI_Type_size( sendtype, &sendsz );
                sendbytes = ( uint64_t )sendcount * sendsz;
            }
            /* MPI_IN_PLACE: sendbytes is initialized to 0 */

            PMPI_Comm_rank( comm, &me );
            if ( me == root )
            {
                PMPI_Comm_size( comm, &N );
                PMPI_Type_size( recvtype, &recvsz );

                for ( i = 0; i < N; ++i )
                {
                    recvbytes += ( uint64_t )recvcounts[ i ] * recvsz;
                }

        #if HAVE( MPI_IN_PLACE )
                if ( sendbuf == MPI_IN_PLACE )
                {
                    recvbytes -= ( uint64_t )recvcounts[ me ] * recvsz;
                }
        #endif
            }


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GATHERV ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GATHERV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Gatherv( sendbuf, sendcount, sendtype, recvbuf, recvcounts, displs, recvtype, root, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Gatherv( sendbuf, sendcount, sendtype, recvbuf, recvcounts, displs, recvtype, root, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_GATHERV,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GATHERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_GATHERV ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_REDUCE ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Reduce )
/**
 * Measurement wrapper for MPI_Reduce
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Reduce( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int count, MPI_Datatype datatype, MPI_Op op, int root, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            sz, me, N;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = root;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Type_size( datatype, &sz );
            PMPI_Comm_rank( comm, &me );
            PMPI_Comm_size( comm, &N );

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf != MPI_IN_PLACE )
      #endif
            {
                sendbytes = ( uint64_t )count * sz;
            }
      #if HAVE( MPI_IN_PLACE )
            else
      #endif
            {
                --N;
            }

            if ( root == me )
            {
                recvbytes = ( uint64_t )N * count * sz;
            }


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Reduce( sendbuf, recvbuf, count, datatype, op, root, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Reduce( sendbuf, recvbuf, count, datatype, op, root, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_REDUCE,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_REDUCE_SCATTER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Reduce_scatter )
/**
 * Measurement wrapper for MPI_Reduce_scatter
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Reduce_scatter( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, SCOREP_MPI_CONST_DECL int* recvcounts, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            i, sz, me, N, count = 0;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Type_size( datatype, &sz );
            PMPI_Comm_rank( comm, &me );
            PMPI_Comm_size( comm, &N );

            for ( i = 0; i < N; i++ )
            {
                count += recvcounts[ i ];
            }

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf == MPI_IN_PLACE )
            {
                sendbytes = ( uint64_t )( count - 1 ) * sz;
                recvbytes = ( uint64_t )( N - 1 ) * recvcounts[ me ] * sz;
            }
            else
      #endif
            {
                sendbytes = ( uint64_t )count * sz;
                recvbytes = ( uint64_t )N * recvcounts[ me ] * sz;
            }


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_SCATTER ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_SCATTER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Reduce_scatter( sendbuf, recvbuf, recvcounts, datatype, op, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Reduce_scatter( sendbuf, recvbuf, recvcounts, datatype, op, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_REDUCE_SCATTER,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_SCATTER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_SCATTER ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_REDUCE_SCATTER_BLOCK ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Reduce_scatter_block )
/**
 * Measurement wrapper for MPI_Reduce_scatter_block
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Reduce_scatter_block( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int recvcount, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            sz, N;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Type_size( datatype, &sz );
            PMPI_Comm_size( comm, &N );

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf == MPI_IN_PLACE )
            {
                --N;
            }
      #endif

            sendbytes = ( uint64_t )N * recvcount * sz;
            recvbytes = sendbytes;


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_SCATTER_BLOCK ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_SCATTER_BLOCK ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Reduce_scatter_block( sendbuf, recvbuf, recvcount, datatype, op, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Reduce_scatter_block( sendbuf, recvbuf, recvcount, datatype, op, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_REDUCE_SCATTER_BLOCK,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_SCATTER_BLOCK ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_SCATTER_BLOCK ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_SCAN ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Scan )
/**
 * Measurement wrapper for MPI_Scan
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Scan( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int count, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            sz, me, N;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = SCOREP_INVALID_ROOT_RANK;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Type_size( datatype, &sz );
            PMPI_Comm_rank( comm, &me );
            PMPI_Comm_size( comm, &N );

      #if HAVE( MPI_IN_PLACE )
            if ( sendbuf == MPI_IN_PLACE )
            {
                sendbytes = ( uint64_t )( N - me - 1 ) * count * sz;
                recvbytes = ( uint64_t )me * count * sz;
            }
            else
      #endif
            {
                sendbytes = ( uint64_t )( N - me ) * count * sz;
                recvbytes = ( uint64_t )( me + 1 ) * count * sz;
            }


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCAN ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCAN ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Scan( sendbuf, recvbuf, count, datatype, op, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Scan( sendbuf, recvbuf, count, datatype, op, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_SCAN,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCAN ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCAN ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_SCATTER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Scatter )
/**
 * Measurement wrapper for MPI_Scatter
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Scatter( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, int root, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            sendsz, recvsz, N, me;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = root;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            PMPI_Comm_rank( comm, &me );

      #if HAVE( MPI_IN_PLACE )
            if ( recvbuf != MPI_IN_PLACE )
      #endif
            {
                if ( me == root )
                {
                    PMPI_Comm_size( comm, &N );
                    PMPI_Type_size( sendtype, &sendsz );
                    sendbytes = ( uint64_t )N * sendcount * sendsz;
                }

                PMPI_Type_size( recvtype, &recvsz );
                recvbytes = ( uint64_t )recvcount * recvsz;
            }
      #if HAVE( MPI_IN_PLACE )
            else
            {
                PMPI_Comm_size( comm, &N );
                PMPI_Type_size( sendtype, &sendsz );
                sendbytes = ( uint64_t )( N - 1 ) * sendcount * sendsz;
                /* recvbytes is initialized to 0 */
            }
      #endif


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCATTER ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCATTER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Scatter( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, root, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Scatter( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, root, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_SCATTER,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCATTER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCATTER ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_SCATTERV ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Scatterv )
/**
 * Measurement wrapper for MPI_Scatterv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Coll.w
 * @note C interface
 * @note Introduced with MPI-1
 * @ingroup coll
 * It wraps the mpi call with an enter and exit event. Additionally, a collective
 * event is generated in between the enter and exit event after the PMPI call.
 */
int
MPI_Scatterv( SCOREP_MPI_CONST_DECL void* sendbuf, SCOREP_MPI_CONST_DECL int* sendcounts, SCOREP_MPI_CONST_DECL int* displs, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, int root, MPI_Comm comm )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;
    uint64_t  start_time_stamp;

    int            sendcount, sendsz, recvsz, me, N, i;
    uint64_t       sendbytes = 0, recvbytes = 0;
    SCOREP_MpiRank root_loc  = root;


    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            sendcount = sendsz = 0;
      #if HAVE( MPI_IN_PLACE )
            if ( recvbuf != MPI_IN_PLACE )
      #endif
            {
                PMPI_Type_size( recvtype, &recvsz );
                recvbytes = ( uint64_t )recvcount * recvsz;
            }
            /* MPI_IN_PLACE: recvbytes is initialized to 0 */

            PMPI_Comm_rank( comm, &me );
            if ( me == root )
            {
                PMPI_Comm_size( comm, &N );
                PMPI_Type_size( sendtype, &sendsz );
                for ( i = 0; i < N; i++ )
                {
                    sendcount += sendcounts[ i ];
                }

        #if HAVE( MPI_IN_PLACE )
                if ( recvbuf == MPI_IN_PLACE )
                {
                    sendcount -= sendcounts[ me ];
                }
        #endif
            }
            sendbytes = ( uint64_t )sendcount * sendsz;


            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCATTERV ] );
            SCOREP_MpiCollectiveBegin();
            start_time_stamp =
                SCOREP_Location_GetLastTimestamp( SCOREP_Location_GetCurrentCPULocation() );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCATTERV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Scatterv( sendbuf, sendcounts, displs, sendtype, recvbuf, recvcount, recvtype, root, comm );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
          #if !defined( SCOREP_MPI_NO_HOOKS )
            if ( SCOREP_IS_MPI_HOOKS_ON )
            {
                SCOREP_Hooks_Post_MPI_Scatterv( sendbuf, sendcounts, displs, sendtype, recvbuf, recvcount, recvtype, root, comm, start_time_stamp, return_val );
            }
          #endif

            SCOREP_MpiCollectiveEnd( SCOREP_MPI_COMM_HANDLE( comm ),
                                     root_loc,
                                     SCOREP_MPI_COLLECTIVE__MPI_SCATTERV,
                                     sendbytes,
                                     recvbytes );
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCATTERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_SCATTERV ] );
        }

        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_IALLGATHER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Iallgather )
/**
 * Measurement wrapper for MPI_Iallgather
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Iallgather call with enter and exit events.
 */
int
MPI_Iallgather( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLGATHER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLGATHER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Iallgather( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLGATHER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLGATHER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IALLGATHERV ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Iallgatherv )
/**
 * Measurement wrapper for MPI_Iallgatherv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Iallgatherv call with enter and exit events.
 */
int
MPI_Iallgatherv( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, SCOREP_MPI_CONST_DECL int* recvcounts, SCOREP_MPI_CONST_DECL int* displs, MPI_Datatype recvtype, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLGATHERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLGATHERV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Iallgatherv( sendbuf, sendcount, sendtype, recvbuf, recvcounts, displs, recvtype, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLGATHERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLGATHERV ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IALLREDUCE ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Iallreduce )
/**
 * Measurement wrapper for MPI_Iallreduce
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Iallreduce call with enter and exit events.
 */
int
MPI_Iallreduce( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int count, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLREDUCE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLREDUCE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Iallreduce( sendbuf, recvbuf, count, datatype, op, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLREDUCE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLREDUCE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IALLTOALL ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Ialltoall )
/**
 * Measurement wrapper for MPI_Ialltoall
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Ialltoall call with enter and exit events.
 */
int
MPI_Ialltoall( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ialltoall( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IALLTOALLV ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Ialltoallv )
/**
 * Measurement wrapper for MPI_Ialltoallv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Ialltoallv call with enter and exit events.
 */
int
MPI_Ialltoallv( SCOREP_MPI_CONST_DECL void* sendbuf, SCOREP_MPI_CONST_DECL int* sendcounts, SCOREP_MPI_CONST_DECL int* sdispls, MPI_Datatype sendtype, void* recvbuf, SCOREP_MPI_CONST_DECL int* recvcounts, SCOREP_MPI_CONST_DECL int* rdispls, MPI_Datatype recvtype, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALLV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALLV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ialltoallv( sendbuf, sendcounts, sdispls, sendtype, recvbuf, recvcounts, rdispls, recvtype, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALLV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALLV ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IALLTOALLW ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Ialltoallw )
/**
 * Measurement wrapper for MPI_Ialltoallw
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Ialltoallw call with enter and exit events.
 */
int
MPI_Ialltoallw( SCOREP_MPI_CONST_DECL void* sendbuf, SCOREP_MPI_CONST_DECL int sendcounts[], SCOREP_MPI_CONST_DECL int sdispls[], SCOREP_MPI_CONST_DECL MPI_Datatype sendtypes[], void* recvbuf, SCOREP_MPI_CONST_DECL int recvcounts[], SCOREP_MPI_CONST_DECL int rdispls[], SCOREP_MPI_CONST_DECL MPI_Datatype recvtypes[], MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALLW ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALLW ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ialltoallw( sendbuf, sendcounts, sdispls, sendtypes, recvbuf, recvcounts, rdispls, recvtypes, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALLW ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IALLTOALLW ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IBARRIER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Ibarrier )
/**
 * Measurement wrapper for MPI_Ibarrier
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Ibarrier call with enter and exit events.
 */
int
MPI_Ibarrier( MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int      event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int      event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int            return_val;
    SCOREP_MpiRank root_loc = SCOREP_INVALID_ROOT_RANK;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBARRIER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBARRIER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ibarrier( comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBARRIER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBARRIER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IBCAST ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Ibcast )
/**
 * Measurement wrapper for MPI_Ibcast
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Ibcast call with enter and exit events.
 */
int
MPI_Ibcast( void* buffer, int count, MPI_Datatype datatype, int root, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBCAST ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBCAST ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ibcast( buffer, count, datatype, root, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBCAST ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IBCAST ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IEXSCAN ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Iexscan )
/**
 * Measurement wrapper for MPI_Iexscan
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Iexscan call with enter and exit events.
 */
int
MPI_Iexscan( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int count, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IEXSCAN ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IEXSCAN ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Iexscan( sendbuf, recvbuf, count, datatype, op, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IEXSCAN ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IEXSCAN ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IGATHER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Igather )
/**
 * Measurement wrapper for MPI_Igather
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Igather call with enter and exit events.
 */
int
MPI_Igather( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, int root, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IGATHER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IGATHER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Igather( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, root, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IGATHER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IGATHER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IGATHERV ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Igatherv )
/**
 * Measurement wrapper for MPI_Igatherv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Igatherv call with enter and exit events.
 */
int
MPI_Igatherv( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, SCOREP_MPI_CONST_DECL int* recvcounts, SCOREP_MPI_CONST_DECL int* displs, MPI_Datatype recvtype, int root, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IGATHERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IGATHERV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Igatherv( sendbuf, sendcount, sendtype, recvbuf, recvcounts, displs, recvtype, root, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IGATHERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IGATHERV ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IREDUCE ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Ireduce )
/**
 * Measurement wrapper for MPI_Ireduce
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Ireduce call with enter and exit events.
 */
int
MPI_Ireduce( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int count, MPI_Datatype datatype, MPI_Op op, int root, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ireduce( sendbuf, recvbuf, count, datatype, op, root, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IREDUCE_SCATTER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Ireduce_scatter )
/**
 * Measurement wrapper for MPI_Ireduce_scatter
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Ireduce_scatter call with enter and exit events.
 */
int
MPI_Ireduce_scatter( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, SCOREP_MPI_CONST_DECL int* recvcounts, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE_SCATTER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE_SCATTER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ireduce_scatter( sendbuf, recvbuf, recvcounts, datatype, op, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE_SCATTER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE_SCATTER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_IREDUCE_SCATTER_BLOCK ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Ireduce_scatter_block )
/**
 * Measurement wrapper for MPI_Ireduce_scatter_block
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Ireduce_scatter_block call with enter and exit events.
 */
int
MPI_Ireduce_scatter_block( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int recvcount, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE_SCATTER_BLOCK ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE_SCATTER_BLOCK ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Ireduce_scatter_block( sendbuf, recvbuf, recvcount, datatype, op, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE_SCATTER_BLOCK ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_IREDUCE_SCATTER_BLOCK ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ISCAN ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Iscan )
/**
 * Measurement wrapper for MPI_Iscan
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Iscan call with enter and exit events.
 */
int
MPI_Iscan( SCOREP_MPI_CONST_DECL void* sendbuf, void* recvbuf, int count, MPI_Datatype datatype, MPI_Op op, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCAN ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCAN ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Iscan( sendbuf, recvbuf, count, datatype, op, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCAN ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCAN ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ISCATTER ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Iscatter )
/**
 * Measurement wrapper for MPI_Iscatter
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Iscatter call with enter and exit events.
 */
int
MPI_Iscatter( SCOREP_MPI_CONST_DECL void* sendbuf, int sendcount, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, int root, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCATTER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCATTER ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Iscatter( sendbuf, sendcount, sendtype, recvbuf, recvcount, recvtype, root, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCATTER ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCATTER ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
#if HAVE( DECL_PMPI_ISCATTERV ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Iscatterv )
/**
 * Measurement wrapper for MPI_Iscatterv
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-3
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Iscatterv call with enter and exit events.
 */
int
MPI_Iscatterv( SCOREP_MPI_CONST_DECL void* sendbuf, SCOREP_MPI_CONST_DECL int* sendcounts, SCOREP_MPI_CONST_DECL int* displs, MPI_Datatype sendtype, void* recvbuf, int recvcount, MPI_Datatype recvtype, int root, MPI_Comm comm, MPI_Request* request )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCATTERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCATTERV ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Iscatterv( sendbuf, sendcounts, displs, sendtype, recvbuf, recvcount, recvtype, root, comm, request );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCATTERV ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_ISCATTERV ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif

#if HAVE( DECL_PMPI_REDUCE_LOCAL ) && !defined( SCOREP_MPI_NO_COLL ) && !defined( MPI_Reduce_local )
/**
 * Measurement wrapper for MPI_Reduce_local
 * @note Auto-generated by wrapgen from template: SCOREP_Mpi_Std.w
 * @note C interface
 * @note Introduced with MPI-2
 * @ingroup coll
 * Triggers an enter and exit event.
 * It wraps the MPI_Reduce_local call with enter and exit events.
 */
int
MPI_Reduce_local( SCOREP_MPI_CONST_DECL void* inbuf, void* inoutbuf, int count, MPI_Datatype datatype, MPI_Op op )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    const int event_gen_active           = SCOREP_MPI_IS_EVENT_GEN_ON;
    const int event_gen_active_for_group = SCOREP_MPI_IS_EVENT_GEN_ON_FOR( SCOREP_MPI_ENABLED_COLL );
    int       return_val;

    if ( event_gen_active )
    {
        SCOREP_MPI_EVENT_GEN_OFF();
        if ( event_gen_active_for_group )
        {
            SCOREP_EnterWrappedRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_LOCAL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_EnterWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_LOCAL ] );
        }
    }

    SCOREP_ENTER_WRAPPED_REGION();
    return_val = PMPI_Reduce_local( inbuf, inoutbuf, count, datatype, op );
    SCOREP_EXIT_WRAPPED_REGION();

    if ( event_gen_active )
    {
        if ( event_gen_active_for_group )
        {
            SCOREP_ExitRegion( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_LOCAL ] );
        }
        else if ( SCOREP_IsUnwindingEnabled() )
        {
            SCOREP_ExitWrapper( scorep_mpi_regions[ SCOREP_MPI_REGION__MPI_REDUCE_LOCAL ] );
        }
        SCOREP_MPI_EVENT_GEN_ON();
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
    return return_val;
}
#endif
/**
 * @}
 */
