/*
 * This file is part of the Score-P software (http://www.score-p.org)
 *
 * Copyright (c) 2009-2012,
 * RWTH Aachen University, Germany
 *
 * Copyright (c) 2009-2012,
 * Gesellschaft fuer numerische Simulation mbH Braunschweig, Germany
 *
 * Copyright (c) 2009-2012, 2015,
 * Technische Universitaet Dresden, Germany
 *
 * Copyright (c) 2009-2012,
 * University of Oregon, Eugene, USA
 *
 * Copyright (c) 2009-2013,
 * Forschungszentrum Juelich GmbH, Germany
 *
 * Copyright (c) 2009-2012,
 * German Research School for Simulation Sciences GmbH, Juelich/Aachen, Germany
 *
 * Copyright (c) 2009-2012,
 * Technische Universitaet Muenchen, Germany
 *
 * This software may be modified and distributed under the terms of
 * a BSD-style license.  See the COPYING file in the package base
 * directory for details.
 */

/**
 * @file
 *
 * @brief SUN f90 compiler PHAT interface.
 * Compiler adapter for the SUN F90 compiler.
 */

#include <config.h>

#include <stdlib.h>

#include <SCOREP_RuntimeManagement.h>
#include <SCOREP_InMeasurement.h>
#include <SCOREP_Types.h>
#include <SCOREP_Events.h>
#include <SCOREP_Mutex.h>

#include "SCOREP_Compiler_Init.h"
#include "scorep_compiler_sun_data.h"


/**
 * This function is called at the entry of each function
 * The call is generated by the SUN f90 compilers
 */
void
phat_enter( char* name,
            int*  id )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    if ( SCOREP_IS_MEASUREMENT_PHASE( PRE ) )
    {
        SCOREP_InitMeasurement();
    }
    if ( !SCOREP_IS_MEASUREMENT_PHASE( WITHIN ) || SCOREP_IsUnwindingEnabled() )
    {
        SCOREP_IN_MEASUREMENT_DECREMENT();
        return;
    }

    if ( *id == SCOREP_INVALID_REGION )
    {
        /* region entered the first time, register region */
        SCOREP_MutexLock( scorep_compiler_region_mutex );
        if ( *id == SCOREP_INVALID_REGION )
        {
            *id = scorep_compiler_register_region( name );
        }
        SCOREP_MutexUnlock( scorep_compiler_region_mutex );
    }

    if ( *id != SCOREP_FILTERED_REGION )
    {
        SCOREP_EnterRegion( *id );
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
}


/**
 * This function is called at the exit of each function
 * The call is generated by the SUN F90 compilers
 */
void
phat_exit( char* name,
           int*  id )
{
    SCOREP_IN_MEASUREMENT_INCREMENT();
    if ( !SCOREP_IS_MEASUREMENT_PHASE( WITHIN ) || SCOREP_IsUnwindingEnabled() )
    {
        SCOREP_IN_MEASUREMENT_DECREMENT();
        return;
    }

    if ( ( *id != SCOREP_INVALID_REGION ) && ( *id != SCOREP_FILTERED_REGION ) )
    {
        SCOREP_ExitRegion( *id );
    }

    SCOREP_IN_MEASUREMENT_DECREMENT();
}
