/*
 * This file is part of the Score-P software (http://www.score-p.org)
 *
 * Copyright (c) 2009-2013,
 *    RWTH Aachen University, Germany
 *
 * Copyright (c) 2009-2013,
 *    Gesellschaft fuer numerische Simulation mbH Braunschweig, Germany
 *
 * Copyright (c) 2009-2016,
 *    Technische Universitaet Dresden, Germany
 *
 * Copyright (c) 2009-2013,
 *    University of Oregon, Eugene, USA
 *
 * Copyright (c) 2009-2013,
 *    Forschungszentrum Juelich GmbH, Germany
 *
 * Copyright (c) 2009-2013,
 *    German Research School for Simulation Sciences GmbH, Juelich/Aachen, Germany
 *
 * Copyright (c) 2009-2013,
 *    Technische Universitaet Muenchen, Germany
 *
 * This software may be modified and distributed under the terms of
 * a BSD-style license.  See the COPYING file in the package base
 * directory for details.
 *
 */

/**
 * @file
 *
 * @brief Types used by metric service.
 */


#ifndef SCOREP_METRIC_TYPES_H
#define SCOREP_METRIC_TYPES_H

#include <stdint.h>

/* *********************************************************************
 * Type definitions
 **********************************************************************/



/**
 * Metric sources to be used in defining a metric member (SCOREP_Definitions_NewMetric()).
 *
 */
typedef enum SCOREP_MetricSourceType
{
    /** PAPI counter. */
    SCOREP_METRIC_SOURCE_TYPE_PAPI   = 0,
    /** Resource usage counter. */
    SCOREP_METRIC_SOURCE_TYPE_RUSAGE = 1,
    /** User metrics. */
    SCOREP_METRIC_SOURCE_TYPE_USER   = 2,
    /** Any other metrics. */
    SCOREP_METRIC_SOURCE_TYPE_OTHER  = 3,
    /** Generated by task profiling. */
    SCOREP_METRIC_SOURCE_TYPE_TASK   = 4,
    /** Plugin metrics. */
    SCOREP_METRIC_SOURCE_TYPE_PLUGIN = 5,
    /** Linux perf metrics */
    SCOREP_METRIC_SOURCE_TYPE_PERF   = 6,

    SCOREP_INVALID_METRIC_SOURCE_TYPE /**< \internal For internal use only. */
} SCOREP_MetricSourceType;

/**
 * Types used to define type of profiling.
 */
typedef enum SCOREP_MetricProfilingType
{
    /** Exclusive values (excludes values from subordinated items) */
    SCOREP_METRIC_PROFILING_TYPE_EXCLUSIVE = 0,
    /** Inclusive values (sum including values from subordinated items ) */
    SCOREP_METRIC_PROFILING_TYPE_INCLUSIVE = 1,
    /** Single value */
    SCOREP_METRIC_PROFILING_TYPE_SIMPLE    = 2,
    /** Maximum values */
    SCOREP_METRIC_PROFILING_TYPE_MAX       = 3,
    /** Minimum values */
    SCOREP_METRIC_PROFILING_TYPE_MIN       = 4,

    SCOREP_INVALID_METRIC_PROFILING_TYPE /**< \internal For internal use only. */
} SCOREP_MetricProfilingType;

/**
 * Types to be used in defining type of metric values (SCOREP_Definitions_NewMetric()).
 * The interface uses UINT64 for all values, the other types should be
 * reinterpreted using a union.
 */
typedef enum SCOREP_MetricValueType
{
    /** 64 bit integer */
    SCOREP_METRIC_VALUE_INT64,
    /** 64 bit unsigned integer */
    SCOREP_METRIC_VALUE_UINT64,
    /** double precision floating point */
    SCOREP_METRIC_VALUE_DOUBLE,

    SCOREP_INVALID_METRIC_VALUE_TYPE /**< \internal For internal use only. */
} SCOREP_MetricValueType;

/**
 * Types to be used in defining metric mode (SCOREP_Definitions_NewMetric()).
 * The mode consists of a timing and a value semantic.
 * The possible value semantics are:
 * <ul>
 * <li>Accumulated for classic counters, e.g. number of floating point
 * operations. While they are stored monotonically increasing in the trace,
 * they are often differentiated as rate over time.</li>
 * <li>Absolute values, e.g. temperature. They are stored as is in the trace
 * and typically also displayed as is.</li>
 * <li>Relative values.</li>
 * </ul>
 * The possible timing semantics are:
 * <ul>
 * <li>Start: The value is valid for the interval from the beginning of the
 * trace to the associated timestamp.</li>
 * <li>Point: The value is only valid for the point in time given by the
 * timestamp.</li>
 * <li>Last: The value is valid for the interval from the previous to the
 * current timestamp.</li>
 * <li>Next: The value is valid for the interval from the current to the
 * next timestamp.</li>
 * </ul>
 */
typedef enum SCOREP_MetricMode
{
    /** Accumulated metric, 'START' timing. */
    SCOREP_METRIC_MODE_ACCUMULATED_START = 0,
    /** Accumulated metric, 'POINT' timing. */
    SCOREP_METRIC_MODE_ACCUMULATED_POINT = 1,
    /** Accumulated metric, 'LAST' timing. */
    SCOREP_METRIC_MODE_ACCUMULATED_LAST  = 2,
    /** Accumulated metric, 'NEXT' timing. */
    SCOREP_METRIC_MODE_ACCUMULATED_NEXT  = 3,

    /** Absolute metric, 'POINT' timing. */
    SCOREP_METRIC_MODE_ABSOLUTE_POINT    = 4,
    /** Absolute metric, 'LAST' timing. */
    SCOREP_METRIC_MODE_ABSOLUTE_LAST     = 5,
    /** Absolute metric, 'NEXT' timing. */
    SCOREP_METRIC_MODE_ABSOLUTE_NEXT     = 6,

    /** Relative metric, 'POINT' timing. */
    SCOREP_METRIC_MODE_RELATIVE_POINT    = 7,
    /** Relative metric, 'LAST' timing. */
    SCOREP_METRIC_MODE_RELATIVE_LAST     = 8,
    /** Relative metric, 'NEXT' timing. */
    SCOREP_METRIC_MODE_RELATIVE_NEXT     = 9,

    SCOREP_INVALID_METRIC_MODE /**< \internal For internal use only. */
} SCOREP_MetricMode;

/**
 * Types to be used in defining metric base (SCOREP_Definitions_NewMetric()).
 *
 */
typedef enum SCOREP_MetricBase
{
    /** Binary base. */
    SCOREP_METRIC_BASE_BINARY  = 0,
    /** Decimal base. */
    SCOREP_METRIC_BASE_DECIMAL = 1,

    SCOREP_INVALID_METRIC_BASE /**< For \internal use only. */
} SCOREP_MetricBase;

/**
 * Properties describing a metric.
 *
 */
typedef struct SCOREP_Metric_Properties
{
    /** Name of the metric. */
    const char*                name;
    /** Long description of the metric. */
    const char*                description;
    /** Type of the metric source (e.g. PAPI). */
    SCOREP_MetricSourceType    source_type;
    /** Metric mode (valid combination of ACCUMULATED|ABSOLUTE|RELATIVE and POINT|START|LAST|NEXT). */
    SCOREP_MetricMode          mode;
    /** Type of the metric value (INT64, UINT64, or DOUBLE). */
    SCOREP_MetricValueType     value_type;
    /** Base of metric values (DECIMAL or BINARY). */
    SCOREP_MetricBase          base;
    /** Exponent to scale metric values (metric_value = value * base^exponent). */
    int64_t                    exponent;
    /** Unit of the metric. */
    const char*                unit;
    /** Profiling type of the metric. */
    SCOREP_MetricProfilingType profiling_type;
} SCOREP_Metric_Properties;

/**
 * Enumeration to define how many threads should record the metrics of a
 * plugin. Used by @ref SCOREP_Metric_Plugin_Info::run_per.
 */
typedef enum SCOREP_MetricPer
{
    /** Metric values are recorded on all threads of all processes */
    SCOREP_METRIC_PER_THREAD = 0,
    /**
     * If processes use multiple threads, the metric is recorded on
     * the main thread of each process.
     */
    SCOREP_METRIC_PER_PROCESS,
    /**
     * Metric values are recorded on a single thread of each node
     * in a parallel program running on multiple nodes (hosts).
     * Nodes are determined by the platform-specific @scorep node
     * identifier.
     */
    SCOREP_METRIC_PER_HOST,
    /**
     * Metric values recorded once within the parallel program.
     * They are recorded on the first node, first process, first thread.
     */
    SCOREP_METRIC_ONCE,

    SCOREP_METRIC_PER_MAX /**< \internal NON-ABI, for internal use only. */
} SCOREP_MetricPer;

/**
 * Enumeration to define the synchronicity type of a metric.
 * Used by @ref SCOREP_Metric_Plugin_Info::sync.
 */
typedef enum SCOREP_MetricSynchronicity
{
    /**
     * The current value of each metric is queried by @scorep whenever an
     * enter/leave event occurs via @ref get_current_value. The plugin
     * must always be able to provide a current value. The plugin provides
     * the value itself, the timestamp is provided by @scorep.
     * This setting is used for metrics that can be measured with minimal
     * runtime costs and updated very frequently.
     */
    SCOREP_METRIC_STRICTLY_SYNC = 0,
    /**
     * The current value of each metric is queried by @scorep whenever an
     * enter/leave event occurs via @ref get_optional_value. Providing a
     * value is optional in case no new value is available in the plugin.
     * The plugin provides the value itself, the timestamp is provided by
     * @scorep.
     * This setting is used for metrics that can be measured with minimal
     * runtime costs but do not necessarily always change.
     */
    SCOREP_METRIC_SYNC,
    /**
     * Metric values are be measured at arbitrary points in time, but
     * are collected at enter/leave events. Whenever an enter/leave event
     * occurs, @scorep queries the plugin via @ref get_all_values for a
     * list of timestamp-value-pairs.
     * This setting can be used for some special cases,
     * #SCOREP_METRIC_ASYNC is usually easier to implement.
     */
    SCOREP_METRIC_ASYNC_EVENT,
    /**
     * Metric values are be measured at arbitrary points in time. All
     * values are collected once at the very end of the execution.
     * @scorep collects the values and associated timestamps via
     * @ref get_all_values.
     * This setting is used for metrics that are recorded on external
     * systems or within a separate thread. While it does require
     * additional memory buffers to store the measurement, it usually
     * reduces the overhead by decoupling the measurement from collection.
     * It is also called post-mortem processing.
     */
    SCOREP_METRIC_ASYNC,

    SCOREP_METRIC_SYNC_TYPE_MAX /**< \internal NON-ABI, for internal use only. */
} SCOREP_MetricSynchronicity;

/**
 * Pair of Score-P timestamp and corresponding metric value (used by
 * asynchronous metrics).
 *
 */
typedef struct SCOREP_MetricTimeValuePair
{
    /** Timestamp in Score-P time! */
    uint64_t timestamp;
    /** Current metric value */
    uint64_t value;
} SCOREP_MetricTimeValuePair;

/**
 * Possible modes of a synchronization point.
 * Express the time when a synchronization happens.
 */
typedef enum SCOREP_MetricSynchronizationMode
{
    /** Synchronization at the beginning of the measurement */
    SCOREP_METRIC_SYNCHRONIZATION_MODE_BEGIN,
    /** Synchronization at the initialization of a multi-process paradigm (e.g., MPI) */
    SCOREP_METRIC_SYNCHRONIZATION_MODE_BEGIN_MPP,
    /** Synchronization at the end of the measurement */
    SCOREP_METRIC_SYNCHRONIZATION_MODE_END,

    SCOREP_METRIC_SYNCHRONIZATION_MODE_MAX /**< \internal NON-ABI, for internal use only. */
} SCOREP_MetricSynchronizationMode;


#endif /* SCOREP_METRIC_TYPES_H */
